/* Octeon file I/O and other OS functions.
 *
 * Copyright (c) 2004, 2005, 2006 Cavium Networks.
 *
 * The authors hereby grant permission to use, copy, modify, distribute,
 * and license this software and its documentation for any purpose, provided
 * that existing copyright notices are retained in all copies and that this
 * notice is included verbatim in any distributions. No written agreement,
 * license, or royalty fee is required for any of the authorized uses.
 * Modifications to this software may be copyrighted by their authors
 * and need not follow the licensing terms described here, provided that
 * the new terms are clearly indicated on the first page of each file where
 * they apply.
 */

#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <stddef.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>
#include <fcntl.h>
#include <stdarg.h>

#include "octeon-uart.h"
#include "octeon-os.h"

struct block
{
  /* Should be multiple of 3 for easier uuencoding.  */
  char data[3*256];
  struct block *next;
};

struct mem_file
{
  char *name;
  struct block *blocks;
  struct block *currblock;
  size_t curroffset;
};

typedef int write_fn (int, const char *, size_t);

static int octeon_os_uart = 0;
uint64_t __boot_desc_addr;
static int octeon_os_console = CONSOLE_TYPE_SIM_MAGIC;
uint32_t octeon_cpu_clock_hz;
static uint64_t rtc_cycles_from_epoch = 0;

/* How many mem_files we have allocated.  */
static size_t n_alloc_files;
/* How many mem files were opened.  */
static int n_mem_files;
/* List of memfiles.  */
static struct mem_file *mem_files;
/* We increment the filedesciptors for memfile starting this number.  */
static const int first_fd = STDERR_FILENO + 1;


/* The size of struct stat is different for N32 and EABI ABI's. Created a 
   struct st which is from EABI ABI to pass it to the simulator magic and 
   later copy the relavant elements of the struct that are used by the 
   application.  */
struct st
{
  dev_t         st_dev;
  ino_t         st_ino;
  mode_t        st_mode;
  nlink_t       st_nlink;
  uid_t         st_uid;
  gid_t         st_gid;
  dev_t         st_rdev;
  int64_t       st_size;
  int64_t       st_atime;
  int64_t       st_spare1;
  int64_t       st_mtime;
  int64_t       st_spare2;
  int64_t       st_ctime;
  int64_t       st_spare3;
  int64_t       st_blksize;
  int64_t       st_blocks;
  int64_t       st_spare4[2];
};

/* This controls whether the console (STDOUT/STDERR, since input is
   not implemented) goes to sim_magic, serial uart, or PCI.  */

void 
octeon_os_set_console(octeon_console_type_t console_type)
{
  octeon_os_console = console_type;
}

void 
octeon_os_set_uart_num (int uart_num)
{
  if (uart_num == 0 || uart_num == 1)
    octeon_os_uart = uart_num;
}

/* simmagic printf() function.  */
void
simprintf (const char *format, ...)
{
  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 6\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
       : : );
}

/* This is the MIPS cache flush function call.  No defines are provided
   by libgloss for 'cache', and CFE doesn't let you flush ranges, so
   we just flush all I & D for every call.  */
void _flush_cache (void)
{
  CVMX_SYNC;
}

/* Open a file descriptor. Using simulator magic.  */
int 
magic_open (const char *buf, int flags, int mode)
{
  register const char *arg1 asm ("$4") = buf;
  register int arg2 asm ("$5") = flags;
  register int arg3 asm ("$6") = mode;
  int ret;

  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 0x9\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	"move %0, $2"
	: "=r"(ret)
	: "r" (arg1), "r" (arg2), "r" (arg3));

  return ret;
}

static void
uuwrite (int desc, struct block *b, struct block *last, size_t lastoff,
	 write_fn *write_console)
{ 
  for (; b; b = b->next)
    {
      size_t len, i, next, lineend;
      char line[60];
      
      len = b == last ? lastoff : sizeof (b->data);
      for (i = 0, next = 0, lineend = 0; i < len; )
	{
	  unsigned w;

	  w = (unsigned char) b->data[i++];
	  w <<= 8;
	  if (i < len)
	    w |= (unsigned char) b->data[i++];
	  w <<= 8;
	  if (i < len)
	    w |= (unsigned char) b->data[i++];

	  line[next++] = (w >> 18 & 0x3f) + 32;
	  line[next++] = (w >> 12 & 0x3f) + 32;
	  line[next++] = (w >> 6 & 0x3f) + 32;
	  line[next++] = (w & 0x3f) + 32;

	  if (next == 60 || i == len)
	    {
	      char length;

	      length = i - lineend + 32;
	      write_console (desc, &length, 1);
	      write_console (desc, line, next);
	      write_console (desc, "\n", 1);

	      lineend = i;
	      next = 0;
	    }
	}
    }
}

static void *
mem_malloc (size_t size)
{
  void *p = malloc (size);
  if (!p)
    {
      __octeon_uart_lock ();
      __octeon_uart_write_raw
	(octeon_os_uart, "memfile: out of memory\r\n", 24);
      __octeon_uart_unlock ();
    }
  return p;
}

void
__octeon_memfile_exit (void)
{
  int i;
  int core;
  char buf[32];

  if (!n_mem_files)
    return;

  core = cvmx_get_core_num ();

  /* At this point memfile already holds the octeon_uart_lock.  So we
     just have to synchronize across cores in memfile.  */
  __octeon_uart_lock ();

  sprintf (buf, "Dumping files on core %d\r\n", core);
  __octeon_uart_write_raw (octeon_os_uart, buf, strlen (buf));

  for (i = 0; i < n_mem_files; i++)
    {
      struct mem_file *f = &mem_files[i];

      __octeon_uart_write_raw (octeon_os_uart, "begin 644 ", 10);
      __octeon_uart_write_raw (octeon_os_uart, f->name, strlen (f->name));
      /* uudecode cannot deal with CRLF here.  */
      __octeon_uart_write_raw (octeon_os_uart, "\n", 1);
      uuwrite (octeon_os_uart, f->blocks, f->currblock, f->curroffset,
	       __octeon_uart_write_raw);
      __octeon_uart_write_raw (octeon_os_uart, "`\nend\n", 6);
    }

  sprintf (buf, "Done dumping files on core %d\r\n", core);
  __octeon_uart_write_raw (octeon_os_uart, buf, strlen (buf));

  __octeon_uart_unlock ();
}

static int
mem_open (const char *name, int flags, int mode)
{
  int fd;
  struct mem_file *f;
  size_t namelen;

  if ((flags & O_ACCMODE) == O_RDONLY)
    {
      errno = EACCES;
      return -1;
    }

  namelen = strlen (name);

  if (n_mem_files == n_alloc_files)
    {
      struct mem_file *new;
      size_t newsize;

      newsize = n_alloc_files ? 2 * n_alloc_files : 10;
      new = mem_malloc (newsize * sizeof (struct mem_file));
      if (!new)
	{
	  errno = ENOMEM;
	  return -1;
	}
      if (n_alloc_files)
	{
	  memcpy (new, mem_files, n_alloc_files * sizeof (struct mem_file));
	  free (mem_files);
	}
      mem_files = new;
      n_alloc_files = newsize;
    }

  f = &mem_files[n_mem_files];
  fd = n_mem_files + first_fd;
  n_mem_files++;

  f->name = mem_malloc (namelen + 1);
  f->blocks = mem_malloc (sizeof (struct block));
  if (!f->name || !f->blocks)
    {
      errno = ENOMEM;
      return -1;
    }

  memcpy (f->name, name, namelen + 1);
  f->currblock = f->blocks;
  f->blocks->next = 0;
  f->curroffset = 0;

  return fd;
}

static int
mem_write (int fd, const void *buf, size_t count)
{
  struct mem_file *f;
  const size_t blocksize = sizeof (f->currblock->data);
  struct block *b;
  size_t totalcount = count;
  int index;

  index = fd - first_fd;
  if (index < 0 || n_mem_files < index)
    {
      errno = EBADF;
      return -1;
    }
  f = &mem_files[index];

  if (f->currblock && blocksize - f->curroffset > 0)
    {
      size_t n = MIN (count, blocksize - f->curroffset);

      memcpy (&f->currblock->data[f->curroffset], buf, n);

      if (n == count)
	{
	  f->curroffset += n;
	  return count;
	}

      count -= n;
      buf += n;
    }

  while (1)
    {
      size_t n;

      b = mem_malloc (sizeof (struct block));
      if (!b)
	{
	  errno = ENOSPC;
	  return -1;
	}
      b->next = NULL;

      f->currblock->next = b;
      f->currblock = b;

      n = MIN (count, blocksize);
      memcpy (f->currblock->data, buf, n);

      if (n == count)
	{
	  f->curroffset = n;
	  return totalcount;
	}

      count -= n;
      buf += n;
    }
  
  return totalcount;
}

static int
mem_read (int fd, void *buf, size_t count)
{
  errno = EINVAL;
  return -1;
}

static off_t 
mem_lseek (int fd, off_t offset, int whence)
{
  errno = EINVAL;
  return -1;
}

static int
mem_close (int fd)
{
  return 0;
}

/* Read bytes from file descriptor. Using simulator magic.  */
static int 
magic_read (int fd, void *buf, size_t len)
{
  register int arg1 asm ("$4") = fd;
  register char *arg2 asm ("$5") = buf;
  register int arg3 asm ("$6") = len;
  int ret;

  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 8\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	"move %0, $2" 
	: "=r" (ret)
	: "r" (arg1), "r" (arg2), "r"(arg3));

  return ret;
}

/* Using simulator magic to write NBYTES from BUF to stdout.  */
static int 
magic_write (int fd, const void *buf, size_t nbytes)
{
  register int arg1 asm ("$4") = fd;
  register const void *arg2 asm ("$5") = buf;
  register int arg3 asm ("$6") = nbytes;
  int ret;

  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 7\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	"move %0, $2"
	: "=r" (ret) 
	: "r" (arg1), "r" (arg2), "r" (arg3));

  return ret;
}

/* Close a file descriptor. Using simulator magic.  */
static int 
magic_close (int fd)
{
  register int arg1 asm ("$4") = fd;
  int ret;

  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 0xA\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	"move %0, $2"
	: "=r"(ret) : "r" (arg1));

  return ret;
}

/* Move read/write pointer. Using simulator simmagic.  */
static off_t 
magic_lseek (int fd, off_t offset, int whence)
{
  register int arg1 asm ("$4") = fd;
  register int arg2 asm ("$5") = offset;
  register int arg3 asm ("$6") = whence;
  int ret;

  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 0xd\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	"move %0, $2"
	: "=r"(ret)
	: "r" (arg1), "r" (arg2), "r" (arg3));
  return ret;
}

int 
_open (const char *buf, int flags, int mode)
{
  if (octeon_os_console == CONSOLE_TYPE_SIM_MAGIC)
      return magic_open (buf, flags, mode);

  return mem_open (buf, flags, mode);
}

int
_read (int fd, void *buf, size_t count)
{
  if (fd == STDOUT_FILENO || fd == STDERR_FILENO)
    {
      errno = EINVAL;
      return -1;
    }

  /* ??? It is weird to check console type for non-console write.  */
  if (octeon_os_console == CONSOLE_TYPE_SIM_MAGIC)
    return magic_read (fd, buf, count);

  return mem_read (fd, buf, count);
}


/* Defined in crt0.S for link ordering reasons */
extern int (*__cvmx_pci_console_write_ptr)(int, char *, int);
/* Used by printf, etc for stdout output.  Redirect the output to
   simulator magic, uart or PCI console  */

int
_write (int fd, const void *buf, size_t nbytes)
{
  if (fd == STDIN_FILENO)
    {
      errno = EINVAL;
      return -1;
    }

  /* Console write.  */
  if (fd == STDOUT_FILENO || fd == STDERR_FILENO)
    {
      if (octeon_os_console == CONSOLE_TYPE_PCI)
	{
	  if (__cvmx_pci_console_write_ptr)
	    return __cvmx_pci_console_write_ptr(fd, (void *) buf, nbytes);
	  else
	    return -1;
	}
      if (octeon_os_console == CONSOLE_TYPE_UART)
	{
	  __octeon_uart_write (octeon_os_uart, buf, nbytes);
	  return nbytes;
	}
    }

  /* ??? It is strange to check console type for non-console write.  */
  if (octeon_os_console == CONSOLE_TYPE_SIM_MAGIC)
    {
      magic_write (fd, buf, nbytes);
      return nbytes;
    }

  return mem_write (fd, buf, nbytes);
}

int
_close (int fd)
{
  if (fd == STDIN_FILENO || fd == STDOUT_FILENO || fd == STDERR_FILENO)
    return 0;

  if (octeon_os_console == CONSOLE_TYPE_SIM_MAGIC)
    return magic_close (fd);

  return mem_close (fd);
}

/* Get status of a file. Since we have no file system, using simulator magic
   to initialize the struct stat.  */
int 
_stat (const char *path, struct stat *sbuf)
{
  struct st local_st;
  register int status asm ("$2");
  register const char *arg1 asm ("$4") = path;
  CVMX_SYNC;
  asm volatile (
	"add $25, $0, 0xB\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	: "=r" (status)
	: "r" (arg1), "r" (&local_st)
	: "$5", "$6");
  sbuf->st_size = local_st.st_size;
  sbuf->st_ino = local_st.st_ino;
  sbuf->st_mode = local_st.st_mode;
  return status;
}

/* Get status of a file. Since we have no file system, using simulator magic
   to initialize the struct stat.  */
int 
_fstat (int fd, struct stat *st)
{
  struct st local_st;
  register int status asm ("$2");
  register int arg1 asm ("$4") = fd;

  CVMX_SYNC;
    
  /* Fake out calls for standard streams */
  if (fd < 3)
    {
      st->st_mode = S_IFCHR;
      return (0);
    }
  asm volatile (
	"add $25, $0, 0xC\n"
	"dli $15,0x8000000feffe0000\n"
	"dadd $24, $31, $0\n"
	"dadd  $4, %[fd], $0\n"
	"dadd  $5, %[buf], $0\n"
	"jalr $15\n"
	"dadd $31, $24, $0\n"
	: "=r" (status)
	: [fd] "r" (arg1), [buf] "r" (&local_st)
	: "$5", "$6");

  st->st_size = local_st.st_size;
  st->st_ino = local_st.st_ino;
  st->st_mode = local_st.st_mode;
  return status;
}

/* Used by newlib as mentioned in the libgloss porting documentation.  */
int 
_isatty (int fd)
{
  return (1);
}

off_t 
_lseek (int fd, off_t offset, int whence)
{
  if (fd == STDIN_FILENO || fd == STDOUT_FILENO || fd == STDERR_FILENO)
    {
      errno = EINVAL;
      return -1;
    }

  /* ??? It is weird to check console type for non-console write.  */
  if (octeon_os_console == CONSOLE_TYPE_SIM_MAGIC)
    return magic_lseek (fd, offset, whence);

  return mem_lseek (fd, offset, whence);
}


/* Heap address and size are defined in boot_elf.h.  */
uint64_t  boot_heap_base;
uint64_t  boot_heap_end;

/* Changes heap size. Get NBYTES more from RAM.  */ 
void *
_sbrk (ptrdiff_t nbytes)
{
  CVMX_SYNC;
  void *tmp;
  if ((boot_heap_base) + nbytes > (boot_heap_end))
    {
    return ((char *)-1);
    }
  else
    {
      tmp = (void *) (long) boot_heap_base;
      boot_heap_base += nbytes;
      return (tmp);
    }
}

/* Used by newlib. We don't have any file system, so creating a process
   is not allowed just return with an error message.  */
int 
_fork ()
{
  errno = EAGAIN;
  return (-1);
}

/* Used by newlib. As per libgloss porting document, returning any value
   greater than 1 doesn't effect anything in newlib because there is
   no process control.  */
#define __MYPID 1
int 
_getpid ()
{
  return __MYPID;
}

/* Used by newlib. There is no process created, waiting for a process to
   terminate does not apply.  */ 
int 
_wait (int *status)
{
  errno = ECHILD;
  return (-1);
}

/* Used by newlib. As per libgloss porting document, kill() doesn't apply
   in an enviornment with no process control, it just exits.  */
int 
_kill (int pid, int sig)
{
  if (pid == __MYPID)
    _exit (sig);

  return 0;
}

/* Used by newlib. File system is not supported, return an error 
   while making a hard link.  */
int 
_link (const char *old, const char *new)
{
  errno = EMLINK;
  return -1;
}

/* Used by newlib. File system is not supported, return an error 
   while removing a hard link.  */
int 
_unlink (const char *name)
{
  errno = ENOENT;
  return -1;
}

/* Return the elapsed time in struct timeval TV.  */
int 
_gettimeofday (struct timeval *tv, struct timezone *tz)              
{
  uint64_t cycle;
  /* Get cycle count.  */
  CVMX_RDHWR (cycle, 31);
  cycle += rtc_cycles_from_epoch;
  if (tv)
    {
      tv->tv_sec = cycle / octeon_cpu_clock_hz;
      tv->tv_usec = (cycle % octeon_cpu_clock_hz) / (octeon_cpu_clock_hz / (1000 * 1000));
      return 0;
    }
  else
    return -1;
}

/* Get Process Times, P1003.1b-1993, p. 92.  */
struct tms {
  int tms_utime;              /* user time */
  int tms_stime;              /* system time */
  int tms_cutime;             /* user time, children */
  int tms_cstime;             /* system time, children */
};

/* Return an error for process consumption time as file I/O is not 
   supported.  */  
int 
_times (struct tms *buf)
{
  return -1;
}

/* Return number of cycles taken at a fixed clock speed.  */
time_t 
time (time_t *t)
{
  uint64_t cycle;
  /* Get cycle count.  */
  CVMX_RDHWR (cycle, 31);
  cycle += rtc_cycles_from_epoch;
  if (t)
    *t = (time_t)cycle / octeon_cpu_clock_hz;

  return ((time_t)cycle / octeon_cpu_clock_hz);
}

/* Set the time as passed in struct timeval tv */
int 
settimeofday (const struct timeval *tv, const struct timezone *tz)
{
  uint64_t cycle, tv_cycles;
  /* Get cycle count.  */
  CVMX_RDHWR (cycle, 31);

  if (tv)
    {
      tv_cycles = (uint64_t)tv->tv_sec * octeon_cpu_clock_hz + ((uint64_t)tv->tv_usec * (octeon_cpu_clock_hz / (1000 * 1000))) % octeon_cpu_clock_hz;
    
      rtc_cycles_from_epoch = tv_cycles - cycle;
      return 0;
    }
  else
    {
      return -1;
    }
}
