/*=============================================================================
//
//      hal_diag.c
//
//      HAL diagnostic output code
//
//=============================================================================
//####COPYRIGHTBEGIN####
//                                                                          
// -------------------------------------------                              
// The contents of this file are subject to the Red Hat eCos Public License 
// Version 1.1 (the "License"); you may not use this file except in         
// compliance with the License.  You may obtain a copy of the License at    
// http://www.redhat.com/                                                   
//                                                                          
// Software distributed under the License is distributed on an "AS IS"      
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the 
// License for the specific language governing rights and limitations under 
// the License.                                                             
//                                                                          
// The Original Code is eCos - Embedded Configurable Operating System,      
// released September 30, 1998.                                             
//                                                                          
// The Initial Developer of the Original Code is Red Hat.                   
// Portions created by Red Hat are                                          
// Copyright (C) 1998, 1999, 2000 Red Hat, Inc.                             
// All Rights Reserved.                                                     
// -------------------------------------------                              
//                                                                          
//####COPYRIGHTEND####
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   cvanschaik, nickg, gthomas
// Contributors:nickg, gthomas
// Date:        1998-03-02
// Purpose:     HAL diagnostic output
// Description: Implementations of HAL diagnostic output support.
//
//####DESCRIPTIONEND####
// Modification : adaptation to EVMCAN (not tested)
//===========================================================================*/

#include <pkgconf/hal.h>
#include <pkgconf/hal_arm_evmcan.h>        // board specifics

#include <cyg/infra/cyg_type.h>         // base types
#include <cyg/infra/cyg_trac.h>         // tracing macros
#include <cyg/infra/cyg_ass.h>          // assertion macros

#include <cyg/hal/hal_arch.h>           // basic machine info
#include <cyg/hal/hal_intr.h>           // interrupt macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_diag.h>
#include <cyg/hal/drv_api.h>
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_misc.h>           // Helper functions

/*---------------------------------------------------------------------------*/
/* From serial_16550.h */
#if CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD==9600
#define CYG_DEVICE_SERIAL_RS232_BAUD        65
#endif
#if CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD==19200
#define CYG_DEVICE_SERIAL_RS232_BAUD        33
#endif
#if CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD==38400
#define CYG_DEVICE_SERIAL_RS232_BAUD        16
#endif
#if CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL_BAUD==115200
#define CYG_DEVICE_SERIAL_RS232_BAUD        5
#endif

#ifndef CYG_DEVICE_SERIAL_RS232_BAUD
#error Missing/incorrect serial baud rate defined - CDL error?
#endif

/****************************************************************************
* USART Module structure
****************************************************************************/
typedef struct
{
    cyg_uint32  PER;               /* PIO Enable Register            */
    cyg_uint32  PDR;               /* PIO Disable Register           */
    cyg_uint32  PSR;               /* PIO Status Register            */
    cyg_uint32  ReservedA;
    cyg_uint32  OER;               /* Output Enable Register         */
    cyg_uint32  ODR;               /* Output Disable Register        */
    cyg_uint32  OSR;               /* Output Status Register         */
    cyg_uint32  ReservedB[5];
    cyg_uint32  SODR;              /* Set Output Data Register       */
    cyg_uint32  CODR;              /* Clear Output Data Register     */
    cyg_uint32  ODSR;              /* Output Data Status Register    */ 
    cyg_uint32  PDSR;              /* Pin Data Status Register       */
    cyg_uint32  MDER;              /* Multi-Driver Enable Register   */
    cyg_uint32  MDDR;              /* Multi-Driver Disable Register  */
    cyg_uint32  MDSR;              /* Multi-Driver Status Register   */
    cyg_uint32  ReservedC;
    cyg_uint32  ECR;               /* Enable Clock Register          */
    cyg_uint32  DCR;               /* Disable Clock Register         */
    cyg_uint32  PMSR;              /* Power Managt Status Register   */
    cyg_uint32  ReservedD;
    cyg_uint32  CR;                /* Control Register               */
    cyg_uint32  MR;                /* Mode Register                  */
    cyg_uint32  ReservedE[2];
    cyg_uint32  SR;                /* Status Register                */
    cyg_uint32  IER;               /* Interrupt Enable Register      */
    cyg_uint32  IDR;               /* Interrupt Disable Register     */
    cyg_uint32  IMR;               /* Interrupt Mask Register        */
    cyg_uint32  RHR;               /* Receive Holding Register       */
    cyg_uint32  THR;               /* Transmit Holding Register      */
    cyg_uint32  BRGR;              /* Baud Rate Generator Register   */
    cyg_uint32  RTOR;              /* Receiver Timeout Register      */
    cyg_uint32  TTGR;              /* Transmitter Timeguard Register */
} EASYCAN_RTOUSART_REGS;


/****************************************************************************
* PIO controller block PER, PDR, PSR, OER, ODR, OSR, SODR, CODR, ODSR,
*                      PDSR, MDER, MDDR, MDSR, SR, IER, IDR, IMR
****************************************************************************/
#define SCK                (0x01 << 16)  // SCK
#define TXD                (0x01 << 17)  // TXD
#define RXD                (0x01 << 18)  // RXD

/****************************************************************************
* PMC controller block ECR, DCR, PMSR
****************************************************************************/
#define PIO                (0x01 << 0)   // PIO
#define USART              (0x01 << 1)   // USART

/****************************************************************************
* Control register  CR
****************************************************************************/
#define SWRST              (0x01 << 0)   // USART software reset
#define RSTRX              (0x01 << 2)   // Reset receiver
#define RSTTX              (0x01 << 3)   // Reset transmitter
#define RXEN               (0x01 << 4)   // Receiver enable
#define RXDIS              (0x01 << 5)   // Receiver disable
#define TXEN               (0x01 << 6)   // Transmitter enable
#define TXDIS              (0x01 << 7)   // Transmitter disable
#define RSTSTA             (0x01 << 8)   // Reset status bits
#define STTBRK             (0x01 << 9)   // Start break
#define STPBRK             (0x01 << 10)  // Stop break
#define STTTO              (0x01 << 11)  // Start timeout
#define SENDA              (0x01 << 12)  // Send address

/****************************************************************************
* Mode register  MR
****************************************************************************/
#define USCLKS             (0x03 << 4)   // Clock selection mask
#define USCLKS_MCKI        (0x00 << 4)   // Internal clock MCKI
#define USCLKS_MCKI_8      (0x01 << 4)   // Internal clock MCKI/8
#define USCLKS_SCK         (0x02 << 4)   // External clock

#define CHRL               (0x03 << 6)   // Byte length
#define CHRL_5             (0x00 << 6)   // Five bits
#define CHRL_6             (0x01 << 6)   // Six bits
#define CHRL_7             (0x02 << 6)   // Seven bits
#define CHRL_8             (0x03 << 6)   // Heigh bits

#define SYNC               (0x01 << 8)   // Synchronous mode
#define ASYNC              (0x00 << 8)   // Asynchronous mode

#define PAR                (0x07 << 9)   // Parity mode
#define PAR_EVEN           (0x00 << 9)   // Even parity
#define PAR_ODD            (0x01 << 9)   // Odd parity
#define PAR_SPACE          (0x02 << 9)   // Space parity (forced to 0)
#define PAR_MARK           (0x03 << 9)   // Mark parity (forced to 1)
#define PAR_NO             (0x04 << 9)   // No parity
#define PAR_MULTIDROP      (0x06 << 9)   // Multi drop parity
 
#define IDLE               (0x01 << 10)  // Idle (J1708 Protocol)
#define IDLEFLAG           (0x01 << 11)  // Idle Flag (J1708 Protocol)

#define NBSTOP             (0x03 << 12)  // Stop bit number
#define NBSTOP_1           (0x00 << 12)  // 1 stop bit
#define NBSTOP_15          (0x01 << 12)  // 1.5 stop bit
#define NBSTOP_2           (0x02 << 12)  // 2 stop bit

#define CHMODE             (0x03 << 14)  // Channel mode
#define CHMODE_NORMAL      (0x00 << 14)  // Normal channel
#define CHMODE_AUTO        (0x01 << 14)  // Automatic echo channel
#define CHMODE_LOCAL       (0x02 << 14)  // Local loopback channel
#define CHMODE_REMOTE      (0x03 << 14)  // Remote loopback channel

#define MODE9              (0x01 << 17)  // 9 bit mode
#define MODE8              (0x00 << 17)  // 8 bit mode
#define CLKO               (0x01 << 18)  // Clock output
#define CLKI               (0x00 << 18)  // Clock input

/****************************************************************************
* Channel Status and Interrupt registers  IER, IDR, IMR, SR
****************************************************************************/
#define RXRDY              (0x01 << 0)   // Receiver ready
#define TXRDY              (0x01 << 1)   // Transmitter ready
#define RXBRK              (0x01 << 2)   // Receiver break
#define ENDRX              (0x01 << 3)   // End of receiver PDC transfer
#define ENDTX              (0x01 << 4)   // End of transmitter PDC transfer
#define USOVRE             (0x01 << 5)   // Overrun error
#define FRAME              (0x01 << 6)   // Framing error
#define PARE               (0x01 << 7)   // Parity error
#define TIMEOUT            (0x01 << 8)   // Receiver time out
#define TXEMPTY            (0x01 << 9)   // Transmitter empty

/****************************************************************************
* Time out 
****************************************************************************/
#define TRANSMIT_TIMEOUT   0xFF
#define RECEIVE_TIMEOUT    0xFF
#define US_BDR_9600        1


#define EASYCAN_RTOUSART0_BASE        0xFFFA8000
#define EASYCAN_RTOUSART1_BASE        0xFFFAC000



//-----------------------------------------------------------------------------
typedef struct {
    volatile EASYCAN_RTOUSART_REGS* base;
    cyg_int32 msec_timeout;
    int isr_vector;
} channel_data_t;

//-----------------------------------------------------------------------------

static void
cyg_hal_plf_serial_init_channel(channel_data_t* __ch_data)
{
    volatile EASYCAN_RTOUSART_REGS *RTOusart = __ch_data->base;
	cyg_uint16 loop_count;

   /* Switch On USART */
   RTOusart->ECR = (USART | PIO);

   /* Reset Error Flags Rx & Tx State Machine */
   RTOusart->CR = (RSTRX | RSTTX);
   RTOusart->CR =  RSTSTA;

	for (loop_count = 1000 ; loop_count > 0 ; loop_count--);

	// Disable port interrupts while changing hardware
	RTOusart->IDR = 0xFFFFFFFF;

	/* Initialize the channel */
	/* Tx and Rx as peripheral */  
      RTOusart->PDR = (TXD | RXD);

    // Set databits, stopbits and parity.
    /* Set Mode */
    RTOusart->MR =CHRL_8|PAR_NO|NBSTOP_1|USCLKS_MCKI;

	// Set baud rate.
	RTOusart->BRGR = CYG_DEVICE_SERIAL_RS232_BAUD;
	/* enable Tx and Rx */
	RTOusart->CR = TXEN|RXEN;
	for (loop_count = 1000 ; loop_count > 0 ; loop_count--);
}

void
cyg_hal_plf_serial_putc(channel_data_t* __ch_data, char c)
{
    volatile EASYCAN_RTOUSART_REGS *RTOusart = __ch_data->base;
    CYGARC_HAL_SAVE_GP();

	while ((RTOusart->SR & TXRDY) == 0);

	RTOusart->THR = c;

    CYGARC_HAL_RESTORE_GP();
}

static cyg_bool
cyg_hal_plf_serial_getc_nonblock(channel_data_t* __ch_data, cyg_uint8* ch)
{
    volatile EASYCAN_RTOUSART_REGS *RTOusart = __ch_data->base;

    if ((RTOusart->SR & RXRDY)  == 0)
        return false;

	*ch = RTOusart->RHR;

    return true;
}

cyg_uint8
cyg_hal_plf_serial_getc(channel_data_t* __ch_data)
{
    volatile EASYCAN_RTOUSART_REGS *RTOusart = __ch_data->base;
    cyg_uint8 ch;
    CYGARC_HAL_SAVE_GP();

	while ((RTOusart->SR & RXRDY) == 0) ;   // Wait for char
	ch = RTOusart->RHR;

    CYGARC_HAL_RESTORE_GP();
    return ch;
}

#if defined(CYGSEM_HAL_VIRTUAL_VECTOR_DIAG) \
    || defined(CYGPRI_HAL_IMPLEMENTS_IF_SERVICES)

static channel_data_t evmcan_ser_channels[2] = {
    { (EASYCAN_RTOUSART_REGS *)EASYCAN_RTOUSART0_BASE, 1000, CYGNUM_HAL_INTERRUPT_USART0 },
    { (EASYCAN_RTOUSART_REGS *)EASYCAN_RTOUSART1_BASE, 1000, CYGNUM_HAL_INTERRUPT_USART1 }
};

static void
cyg_hal_plf_serial_write(void* __ch_data, const cyg_uint8* __buf, 
                         cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        cyg_hal_plf_serial_putc(__ch_data, *__buf++);

    CYGARC_HAL_RESTORE_GP();
}

static void
cyg_hal_plf_serial_read(void* __ch_data, cyg_uint8* __buf, cyg_uint32 __len)
{
    CYGARC_HAL_SAVE_GP();

    while(__len-- > 0)
        *__buf++ = cyg_hal_plf_serial_getc(__ch_data);

    CYGARC_HAL_RESTORE_GP();
}

cyg_bool
cyg_hal_plf_serial_getc_timeout(void* __ch_data, cyg_uint8* ch)
{
    int delay_count;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    cyg_bool res;
    CYGARC_HAL_SAVE_GP();

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        res = cyg_hal_plf_serial_getc_nonblock(__ch_data, ch);
        if (res || 0 == delay_count--)
            break;
        
        CYGACC_CALL_IF_DELAY_US(100);
    }

    CYGARC_HAL_RESTORE_GP();
    return res;
}

static int
cyg_hal_plf_serial_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    static int irq_state = 0;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    int ret = 0;
    CYGARC_HAL_SAVE_GP();

    switch (__func) {
    case __COMMCTL_IRQ_ENABLE:
        irq_state = 1;

		chan->base->IDR = 0xFFFFFFFF;
		chan->base->IER = RXRDY; 
        HAL_INTERRUPT_UNMASK(chan->isr_vector);
        break;
    case __COMMCTL_IRQ_DISABLE:
        ret = irq_state;
        irq_state = 0;

		chan->base->IDR = 0xFFFFFFFF;
        HAL_INTERRUPT_MASK(chan->isr_vector);
        break;
    case __COMMCTL_DBG_ISR_VECTOR:
        ret = chan->isr_vector;
        break;
    case __COMMCTL_SET_TIMEOUT:
    {
        va_list ap;

        va_start(ap, __func);

        ret = chan->msec_timeout;
        chan->msec_timeout = va_arg(ap, cyg_uint32);

        va_end(ap);
    }        
    default:
        break;
    }
    CYGARC_HAL_RESTORE_GP();
    return ret;
}

static int
cyg_hal_plf_serial_isr(void *__ch_data, int* __ctrlc, 
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    int res = 0;
    channel_data_t* chan = (channel_data_t*)__ch_data;
    char c;
    CYGARC_HAL_SAVE_GP();

    cyg_drv_interrupt_acknowledge(chan->isr_vector);

    *__ctrlc = 0;
    if ((chan->base->SR & RXRDY) != 0) {

		c = chan->base->RHR;
        if( cyg_hal_is_break( &c , 1 ) )
            *__ctrlc = 1;

        res = CYG_ISR_HANDLED;
    }

    CYGARC_HAL_RESTORE_GP();
    return res;
}

static void
cyg_hal_plf_serial_init(void)
{
    hal_virtual_comm_table_t* comm;
    int cur = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);

    // Disable interrupts.
    HAL_INTERRUPT_MASK(evmcan_ser_channels[0].isr_vector);
    HAL_INTERRUPT_MASK(evmcan_ser_channels[1].isr_vector);

    // Init channels
    cyg_hal_plf_serial_init_channel(&evmcan_ser_channels[0]);
    cyg_hal_plf_serial_init_channel(&evmcan_ser_channels[1]);

    // Setup procs in the vector table

    // Set channel 0
    CYGACC_CALL_IF_SET_CONSOLE_COMM(0);
    comm = CYGACC_CALL_IF_CONSOLE_PROCS();
    CYGACC_COMM_IF_CH_DATA_SET(*comm, &evmcan_ser_channels[0]);
    CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
    CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
    CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
    CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
    CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
    CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
    CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);

    // Set channel 1
    CYGACC_CALL_IF_SET_CONSOLE_COMM(1);
    comm = CYGACC_CALL_IF_CONSOLE_PROCS();
    CYGACC_COMM_IF_CH_DATA_SET(*comm, &evmcan_ser_channels[1]);
    CYGACC_COMM_IF_WRITE_SET(*comm, cyg_hal_plf_serial_write);
    CYGACC_COMM_IF_READ_SET(*comm, cyg_hal_plf_serial_read);
    CYGACC_COMM_IF_PUTC_SET(*comm, cyg_hal_plf_serial_putc);
    CYGACC_COMM_IF_GETC_SET(*comm, cyg_hal_plf_serial_getc);
    CYGACC_COMM_IF_CONTROL_SET(*comm, cyg_hal_plf_serial_control);
    CYGACC_COMM_IF_DBG_ISR_SET(*comm, cyg_hal_plf_serial_isr);
    CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_getc_timeout);

    // Restore original console
    CYGACC_CALL_IF_SET_CONSOLE_COMM(cur);
}

void
cyg_hal_plf_comms_init(void)
{
    static int initialized = 0;

    if (initialized)
        return;

    initialized = 1;

    cyg_hal_plf_serial_init();
}
#endif // CYGSEM_HAL_VIRTUAL_VECTOR_DIAG || CYGPRI_HAL_IMPLEMENTS_IF_SERVICES

/*---------------------------------------------------------------------------*/

#ifdef CYGHWR_HAL_ARM_evmcan_DIAG_LEDS
// Control the LEDs PP0-PP3. This requires the jumpers on pins 9-16 to
// be set on LK11, thus preventing the use of the parallel port.

#define CYG_DEVICE_PARALLEL_DATA 0x0d800040

void
hal_diag_led(int n)
{
    HAL_WRITE_UINT8(CYG_DEVICE_PARALLEL_DATA, (n & 0xf) <<4);
}
#endif // CYGHWR_HAL_ARM_evmcan_DIAG_LEDS


//=============================================================================
// Compatibility with older stubs
//=============================================================================

#ifndef CYGSEM_HAL_VIRTUAL_VECTOR_DIAG

#ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS
#include <cyg/hal/hal_stub.h>           // cyg_hal_gdb_interrupt
#endif

#if CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL==0
// This is the base address of the 0-channel
#define CYG_DEV_SERIAL_BASE      EASYCAN_RTOUSART0_BASE
#define CYG_DEVICE_SERIAL_INT    CYGNUM_HAL_INTERRUPT_USART0
#else
// This is the base address of the 1-channel
#define CYG_DEV_SERIAL_BASE      EASYCAN_RTOUSART1_BASE
#define CYG_DEVICE_SERIAL_INT    CYGNUM_HAL_INTERRUPT_USART1
#endif

static channel_data_t evmcan_ser_channel = {
    (EASYCAN_RTOUSART_REGS*)CYG_DEV_SERIAL_BASE, 0, 0 
};

// Assumption: all diagnostic output must be GDB packetized unless this is a ROM (i.e.
// totally stand-alone) system.

#if defined(CYG_HAL_STARTUP_ROM) || defined(CYGDBG_HAL_DIAG_DISABLE_GDB_PROTOCOL)
#define HAL_DIAG_USES_HARDWARE
#endif

#ifndef HAL_DIAG_USES_HARDWARE
#if (CYGNUM_HAL_VIRTUAL_VECTOR_CONSOLE_CHANNEL != CYGNUM_HAL_VIRTUAL_VECTOR_DEBUG_CHANNEL)
#define HAL_DIAG_USES_HARDWARE
#endif
#endif

#ifdef HAL_DIAG_USES_HARDWARE

void hal_diag_init(void)
{
    static int init = 0;
    char *msg = "\n\rARM eCos\n\r";

    if (init++) return;

    cyg_hal_plf_serial_init_channel(&evmcan_ser_channel);

    while (*msg) cyg_hal_plf_serial_putc(&evmcan_ser_channel, *msg++);
}

#ifdef DEBUG_DIAG
#if defined(CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS)
#define DIAG_BUFSIZE 32
#else
#define DIAG_BUFSIZE 2048
#endif
static char diag_buffer[DIAG_BUFSIZE];
static int diag_bp = 0;
#endif

void hal_diag_write_char(char c)
{
    hal_diag_init();

    cyg_hal_plf_serial_putc(&evmcan_ser_channel, c);

#ifdef DEBUG_DIAG
    diag_buffer[diag_bp++] = c;
    if (diag_bp == DIAG_BUFSIZE) diag_bp = 0;
#endif
}

void hal_diag_read_char(char *c)
{
    *c = cyg_hal_plf_serial_getc(&evmcan_ser_channel);
}

#else // HAL_DIAG relies on GDB

// Initialize diag port - assume GDB channel is already set up
void hal_diag_init(void)
{
    if (0) cyg_hal_plf_serial_init_channel(&evmcan_ser_channel); // avoid warning
}

// Actually send character down the wire
static void
hal_diag_write_char_serial(char c)
{
    hal_diag_init();

    cyg_hal_plf_serial_putc(&evmcan_ser_channel, c);
}

static bool
hal_diag_read_serial(char *c)
{
    long timeout = 1000000000;  // A long time...
    while (!cyg_hal_plf_serial_getc_nonblock(&evmcan_ser_channel, c))
        if (0 == --timeout) return false;

    return true;
}

void 
hal_diag_read_char(char *c)
{
    while (!hal_diag_read_serial(c)) ;
}

void 
hal_diag_write_char(char c)
{
    static char line[100];
    static int pos = 0;

    // No need to send CRs
    if( c == '\r' ) return;

    line[pos++] = c;

    if( c == '\n' || pos == sizeof(line) )
    {
        CYG_INTERRUPT_STATE old;

        // Disable interrupts. This prevents GDB trying to interrupt us
        // while we are in the middle of sending a packet. The serial
        // receive interrupt will be seen when we re-enable interrupts
        // later.
        
#ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS
        CYG_HAL_GDB_ENTER_CRITICAL_IO_REGION(old);
#else
        HAL_DISABLE_INTERRUPTS(old);
#endif

        while(1)
        {
            static char hex[] = "0123456789ABCDEF";
            cyg_uint8 csum = 0;
            int i;
            char c1;
        
            hal_diag_write_char_serial('$');
            hal_diag_write_char_serial('O');
            csum += 'O';
            for( i = 0; i < pos; i++ )
            {
                char ch = line[i];
                char h = hex[(ch>>4)&0xF];
                char l = hex[ch&0xF];
                hal_diag_write_char_serial(h);
                hal_diag_write_char_serial(l);
                csum += h;
                csum += l;
            }
            hal_diag_write_char_serial('#');
            hal_diag_write_char_serial(hex[(csum>>4)&0xF]);
            hal_diag_write_char_serial(hex[csum&0xF]);

            // Wait for the ACK character '+' from GDB here and handle
            // receiving a ^C instead.  This is the reason for this clause
            // being a loop.
            if (!hal_diag_read_serial(&c1))
                continue;   // No response - try sending packet again

            if( c1 == '+' )
                break;              // a good acknowledge

#ifdef CYGDBG_HAL_DEBUG_GDB_BREAK_SUPPORT
            cyg_drv_interrupt_acknowledge(CYG_DEVICE_SERIAL_INT);
            if( c1 == 3 ) {
                // Ctrl-C: breakpoint.
                cyg_hal_gdb_interrupt ((target_register_t)__builtin_return_address(0));
                break;
            }
#endif
            // otherwise, loop round again
        }
        
        pos = 0;

        // And re-enable interrupts
#ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS
        CYG_HAL_GDB_LEAVE_CRITICAL_IO_REGION(old);
#else
        HAL_RESTORE_INTERRUPTS(old);
#endif
        
    }
}
#endif

#endif // CYGSEM_HAL_VIRTUAL_VECTOR_DIAG

/*---------------------------------------------------------------------------*/
/* End of hal_diag.c */
