/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's insertText command.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: insertText.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */

mozile.require("mozile.edit");
mozile.require("mozile.test.shared.util");
var name = "mozile.edit.insertText";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testInsertAlternateSpaces"];
	return ["testInsertText", "testReplaceText", "testReplaceEmpty", "testInsertSpaces", "testInsertAlternateSpaces"];
}

/**
 * 
 */
function testInsertText() {
	var command = mozile.edit.insertText;
	assertEquals("Name should be 'InsertText'", "InsertText", command.name);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	var range = new mozile.dom.Range();
	range.setStart(text, 0);
	range.collapse(true);
	var selection = mozile.dom.selection.get();
	selection.removeAllRanges();
	selection.addRange(range);
	//alert(selection.focusNode.nodeName +" "+ selection.focusOffset);
	
	var event = {
		type: "keypress", 
		charCode: 65, 
		ctrlKey: false, 
		metaKey: false
	}
	//var string = "This is     whitespace."; 
	var string = "This is a normal sentence. There is no extra whitespace.";
	var states = new Array();
	
	// Execute
	for(var i=0; i < string.length; i++) {
		event.charCode = string.charCodeAt(i);
		//var state = command.prepare(event);
		//state = command.execute(state);
		var state = command.trigger(event);
		states.push(state);
		assertTrue("Command "+ i +" should have been executed.", state.executed);		
		assertEquals("Check the text node contents after execution "+ i, 
			string.substring(0, i+1) + original, text.data);
	}
	
	//alert(text.data);
	
	// Unexecute
	for(i = states.length - 1; i >= 0; i--) {
		var state = states[i];
		state = command.unexecute(state, true); // use "fresh" mode
		assertFalse("Command "+ i +" should have been unexecuted.", state.executed);
		assertEquals("Check the text node contents after unexecution "+ i, 
			string.substring(0, i) + original, text.data);
	}
	
}

/**
 * 
 */
function testReplaceText() {
	mozile.test.shared.preselect();
	var command = mozile.edit.insertText;

	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	var content = "Text content";

	var selection = mozile.dom.selection.get();
	selection.collapse(text, 5);

	var i = 0;
	var state = command.request({}, false, null, content, text);
	assertTrue("Command "+ i +" should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution "+ i, 
		content, text.data);

	assertEquals("The selection anchor should be in the text node", text, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the text node", text, selection.focusNode);
	assertEquals("The selection focus offset should be content.length", content.length, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);	
	
	// Unexecute
	state = command.unexecute(state, false);
	assertFalse("Command "+ i +" should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution "+ i, 
		original, text.data);

	assertEquals("The selection anchor should be in the text node", text, selection.anchorNode);
	assertEquals("The selection anchor offset should be 5", 5, selection.anchorOffset);
	assertTrue("The selection should be collapsed", selection.isCollapsed);
	
}

/**
 * 
 */
function testReplaceEmpty() {
	mozile.test.shared.preselect();
	var command = mozile.edit.insertText;

	var target = document.getElementById("target");
	var text = mozile.edit.createEmptyToken();
	target.appendChild(text);
	var original = text.data;
	var content = "a";

	var selection = mozile.dom.selection.get();
	selection.collapse(text, 0);

	var i = 0;
	var state = command.request({}, false, null, content);
	assertTrue("Command "+ i +" should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution "+ i, 
		content, text.data);

	assertEquals("The selection anchor should be in the text node", text, selection.anchorNode);
	assertEquals("The selection anchor offset should be content.length", content.length, selection.anchorOffset);
	assertTrue("The selection should not be collapsed", selection.isCollapsed);	
	
	// Unexecute
	state = command.unexecute(state, false);
	assertFalse("Command "+ i +" should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution "+ i, 
		original, text.data);

	assertEquals("The selection anchor should be in the text node", text, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertTrue("The selection should be collapsed", selection.isCollapsed);
	
}

/**
 * Test insertText's handling of spaces if no alternateSpace is provided. The command should insert a single space, and then ignore subsequent spaces.
 */
function testInsertSpaces() {
	mozile.test.shared.preselect();
	var command = mozile.edit.insertText;
	var storeAlt = mozile.alternateSpace;
	mozile.alternateSpace = null;
	var space = " ";
	var target = document.getElementById("target");
	var text = target.firstChild;
	var storeText = text.data;
	var original = "test text";
	text.data = original;
	var selection = mozile.dom.selection.get();
	var parentState = {actions: new Array()};
	var state, test;
	var event = {
		type: "keypress",
		charCode: 32
	}
	
	// Start of node
	var i = 1;
	selection.collapse(text, 0);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", space+original, text.data);
	assertEquals(i +" The offset should be 1", 1, selection.focusOffset);
	test = command.test(event, null, space);
	assertFalse(i +" The test should fail", test);
	command.unexecute(state);
	assertEquals(i+" The text should be restored", original, text.data);
	
	// Middle of word
	i = 2;
	selection.collapse(text, 1);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", 
		original.substring(0,1) + space + original.substring(1), text.data);
	assertEquals(i +" The offset should be 2", 2, selection.focusOffset);
	test = command.test(event, null, space);
	assertFalse(i +" The test should fail", test);
	command.unexecute(state);
	assertEquals(i+" The text should be restored", original, text.data);
	
	// Before space
	i = 3;
	selection.collapse(text, 4);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", 
		original.substring(0,4) + space + original.substring(4), text.data);
	assertEquals(i +" The offset should be 5", 5, selection.focusOffset);
	test = command.test(event, null, space);
	assertFalse(i +" The test should fail", test);
	command.unexecute(state);
	assertEquals(i+" The text should be restored", original, text.data);
	
	// End of node
	i = 4;
	selection.collapse(text, text.data.length);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", original + space, text.data);
	assertEquals(i +" The offset should be at the end", text.data.length, selection.focusOffset);
	test = command.test(event, null, space);
	assertFalse(i +" The test should fail", test);
	command.unexecute(state);
	assertEquals(i+" The text should be restored", original, text.data);
	
	mozile.alternateSpace = storeAlt;
	text.data = storeText;
}

/**
 * Test insertText's handling of spaces if an alternateSpace character is provided. The command should insert one space character, then one alternateSpace character. The rules are somewhat complicated.
 */
function testInsertAlternateSpaces() {
	mozile.test.shared.preselect();
	var command = mozile.edit.insertText;
	var storeAlt = mozile.alternateSpace;
	var alt = "#";
	mozile.alternateSpace = alt;
	var space = " ";
	var target = document.getElementById("target");
	var text = target.firstChild;
	var storeText = text.data;
	var original = "test text";
	text.data = original;
	var selection = mozile.dom.selection.get();
	var parentState = {actions: new Array()};
	var state;
	
	// Start of node
	var i = 1;
	selection.collapse(text, 0);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", alt+original, text.data);
	assertEquals(i +" The offset should be 1", 1, selection.focusOffset);
	i = 2;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", alt+space+original, text.data);
	assertEquals(i +" The offset should be 2", 2, selection.focusOffset);
	i = 3;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", alt+space+alt+original, text.data);
	assertEquals(i +" The offset should be 3", 3, selection.focusOffset);
	i = 4;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", alt+space+alt+space+original, text.data);
	assertEquals(i +" The offset should be 4", 4, selection.focusOffset);
	while(parentState.actions.length) 
		command.unexecute(parentState.actions.pop());
	assertEquals(i+" The text should be restored", original, text.data);
	
	// Middle of word
	i = 5;
	selection.collapse(text, 1);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", 
		original.substring(0,1) + space + original.substring(1), text.data);
	assertEquals(i +" The offset should be 2", 2, selection.focusOffset);
	i = 6;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", 
		original.substring(0,1) + space+alt + original.substring(1), text.data);
	assertEquals(i +" The offset should be 3", 3, selection.focusOffset);
	i = 7;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one space", 
		original.substring(0,1) + space+alt+space + original.substring(1), text.data);
	assertEquals(i +" The offset should be 4", 4, selection.focusOffset);
	while(parentState.actions.length) 
		command.unexecute(parentState.actions.pop());
	assertEquals(i+" The text should be restored", original, text.data);
	
	// Before space
	i = 8;
	selection.collapse(text, 4);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", 
		original.substring(0,4) + alt + original.substring(4), text.data);
	assertEquals(i +" The offset should be 5", 5, selection.focusOffset);
	i = 9;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Replace one alternateSpace with a space+alt", 
		original.substring(0,4) + space+alt + original.substring(4), text.data);
	assertEquals(i +" The offset should be 6", 6, selection.focusOffset);
	i = 10;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", 
		original.substring(0,4) + space+alt+alt + original.substring(4), text.data);
	assertEquals(i +" The offset should be 7", 7, selection.focusOffset);
	i = 11;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Replace one alternateSpace with a space+alt", 
		original.substring(0,4) + space+alt+space+alt + original.substring(4), text.data);
	assertEquals(i +" The offset should be 8", 8, selection.focusOffset);
	while(parentState.actions.length) 
		command.unexecute(parentState.actions.pop());
	assertEquals(i+" The text should be restored", original, text.data);
	
	// End of node
	i = 12;
	selection.collapse(text, 9);
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", 
		original + alt, text.data);
	assertEquals(i +" The offset should be 10", 10, selection.focusOffset);
	i = 13;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Replace one alternateSpace with a space+alt", 
		original + space+alt, text.data);
	assertEquals(i +" The offset should be 11", 11, selection.focusOffset);
	i = 14;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Insert one alternateSpace", 
		original + space+alt+alt, text.data);
	assertEquals(i +" The offset should be 12", 12, selection.focusOffset);
	i = 15;
	state = command.request(parentState, false, null, space);
	assertNotNull(i +" The request should have succeeded", state);
	assertEquals(i+" Replace one alternateSpace with a space+alt", 
		original + space+alt+space+alt, text.data);
	assertEquals(i +" The offset should be 13", 13, selection.focusOffset);
	while(parentState.actions.length) 
		command.unexecute(parentState.actions.pop());
	assertEquals(i+" The text should be restored", original, text.data);
	
	mozile.alternateSpace = storeAlt;
	text.data = storeText;
}



