/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's editing functionality.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: clipboard.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */

mozile.require("mozile.edit");
mozile.require("mozile.edit.rich");
mozile.require("mozile.test.shared.util");
var name = "mozile.edit.clipboard";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	return ["testCut"];
	return ["testCopy", "testCut", "testPasteText", "testPasteNodes", "testPasteTextAndNodes"];
}

/**
 *
 */
function testCopy() {
	var command = mozile.edit.copy;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;

	assertNull("The clipboard should be empty", mozile.edit.clipboard);

	// Copy text
	selection.collapse(text, 5);
	selection.extend(text, 15);
	var state = new mozile.edit.State(command);
	command.execute(state, false);
	assertTrue("The state has been executed", state.executed);

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain text", original.substring(5,15), mozile.edit.clipboard);
	mozile.edit.clipboard = null;

	// Copy nodes
	selection.collapse(target.firstChild, 5);
	selection.extend(target.lastChild, 5);
	var state = new mozile.edit.State(command);
	command.execute(state, false);
	assertTrue("The state has been executed", state.executed);

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain a dofumentFragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, mozile.edit.clipboard.nodeType);
	assertEquals("The clipboard should contain a number oc child nodes", target.childNodes.length, mozile.edit.clipboard.childNodes.length);
	
	mozile.edit.clipboard = null;
}

/**
 *
 */
function testCut() {
	var command = mozile.edit.cut;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;

	assertNull("The clipboard should be empty", mozile.edit.clipboard);

	// Copy text
	selection.collapse(text, 5);
	selection.extend(text, 15);
	var after = original.substring(0,5) + original.substring(15);
	var state = new mozile.edit.State(command);
	command.execute(state, false);
	assertTrue("The state has been executed", state.executed);

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain text", original.substring(5,15), mozile.edit.clipboard);
	assertEquals("The text should be removed", after, text.data);
	mozile.edit.clipboard = null;
	
	state = command.unexecute(state, false);
	assertFalse("The state has been unexecuted", state.executed);
	assertEquals("The text should be restored", original, text.data);

	// Copy nodes
	selection.collapse(target.firstChild, 5);
	selection.extend(target.lastChild, 5);
	after = target.firstChild.data.substring(0,5) + target.lastChild.data.substring(5);
	var length = target.childNodes.length;
	var state = new mozile.edit.State(command);
	command.execute(state, false);
	assertTrue("The state has been executed", state.executed);

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain a dofumentFragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, mozile.edit.clipboard.nodeType);
	assertEquals("The clipboard should contain a number of child nodes", length, mozile.edit.clipboard.childNodes.length);
	assertEquals("The text should be removed", after, text.data);
	
	state = command.unexecute(state, false);
	assertFalse("The state has been unexecuted", state.executed);
	assertEquals("The text should be restored", original, text.data);
	
	mozile.edit.clipboard = null;
}

/**
 *
 */
function testPasteText() {
	var insertNode = mozile.edit.insertNode;
	mozile.edit.insertNode = undefined;

	var command = mozile.edit.paste;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	var after = original.substring(0,5) + original.substring(5,15) + original.substring(15);
	selection.collapse(text, 5);
	selection.extend(text, 15);

	assertNull("The clipboard should be empty", mozile.edit.clipboard);
	mozile.edit.clipboard = original.substring(5,15);

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain text", original.substring(5,15), mozile.edit.clipboard);
	
	var state = command.prepare(null);
	state = command.execute(state, false);
	assertTrue("The state has been executed", state.executed);
	assertEquals("The text should have been pasted", after, text.data);
	assertEquals("The selection should be in text", text, selection.focusNode);
	assertEquals("The selection should be at 15", 15, selection.focusOffset);
	
	state = command.unexecute(state, false);
	assertFalse("The state has been unexecuted", state.executed);
	assertEquals("The text should be restored", original, text.data);
	
	mozile.edit.clipboard = null;
	mozile.edit.insertNode = insertNode;
}

/**
 *
 */
function testPasteNodes() {
	var command = mozile.edit.paste;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	var after1 = original.substring(0,5);
	var after2 = original.substring(15);
	selection.collapse(text, 5);
	selection.extend(text, 15);

	assertNull("The clipboard should be empty", mozile.edit.clipboard);
	var df = document.createDocumentFragment();
	df.appendChild(mozile.dom.createElement("foo"));
	df.appendChild(mozile.dom.createElement("bar"));
	df.appendChild(mozile.dom.createElement("baz"));
	mozile.edit.clipboard = df;

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain a documentFragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, mozile.edit.clipboard.nodeType);
	assertEquals("The clipboard should contain three children", 3, mozile.edit.clipboard.childNodes.length);
	
	var state = command.prepare(null);
	state = command.execute(state, false);
	assertTrue("The state has been executed", state.executed);
	assertEquals("The text should have been split", after1, text.data);
	assertEquals("There should be a foo element", "foo", target.childNodes[1].nodeName.toLowerCase());
	assertEquals("There should be a bar element", "bar", target.childNodes[2].nodeName.toLowerCase());
	assertEquals("There should be a baz element", "baz", target.childNodes[3].nodeName.toLowerCase());
	assertEquals("More text should have been split", after2, target.childNodes[4].data);
	assertEquals("The selection should be in text", target.childNodes[3], selection.focusNode);
	assertEquals("The selection should be at 0", 0, selection.focusOffset);
	
	state = command.unexecute(state, false);
	assertFalse("The state has been unexecuted", state.executed);
	assertEquals("The text should be restored", original, text.data);
	
	mozile.edit.clipboard = null;
}

/**
 *
 */
function testPasteTextAndNodes() {
	var command = mozile.edit.paste;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	var after1 = original.substring(0,5) + "foo";
	var after2 = "bar" + original.substring(15);
	selection.collapse(text, 5);
	selection.extend(text, 15);

	assertNull("The clipboard should be empty", mozile.edit.clipboard);
	var df = document.createDocumentFragment();
	df.appendChild(document.createTextNode("foo"));
	df.appendChild(mozile.dom.createElement("foobar"));
	df.appendChild(document.createTextNode("bar"));
	mozile.edit.clipboard = df;

	assertNotNull("The clipboard should not be empty", mozile.edit.clipboard);
	assertEquals("The clipboard should contain a documentFragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, mozile.edit.clipboard.nodeType);
	assertEquals("The clipboard should contain three children", 3, mozile.edit.clipboard.childNodes.length);
	
	var state = command.prepare(null);
	state = command.execute(state, false);
	assertTrue("The state has been executed", state.executed);
	assertEquals("The text should have been pasted", after1, text.data);
	assertEquals("There should be a foobar element", "foobar", text.nextSibling.nodeName.toLowerCase());
	assertEquals("More text should have been pasted", after2, text.nextSibling.nextSibling.data);
	assertEquals("The selection should be in text", text.nextSibling.nextSibling, selection.focusNode);
	assertEquals("The selection should be at 3", 3, selection.focusOffset);
	
	state = command.unexecute(state, false);
	assertFalse("The state has been unexecuted", state.executed);
	assertEquals("The text should be restored", original, text.data);
	
	mozile.edit.clipboard = null;
}
