/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */

/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's core functionality.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: core.js,v 1.2 2006/08/23 16:55:27 jameso Exp $
 */

var name = "mozile.core";

var testSchema = "../rng/include.rng";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testGetAbsolutePath"];
	return ["testLoad", "testFindRoot", "testLoadCore", "testInfo", "testFindModule", "testLoadModule", "testRequire", "testProvide", "testIsPathAbsolute", "testGetPath", "testGetDirectory", "testGetAbsolutePath", "testAlert", "testDumpError", "testEditAllText", "testEditElement", "testEditElements", "testEditDocument", "testUseSchema"];
}

/**
 * 
 */
function testLoad() {
	var content = mozile.load(mozile.test.root + "/tests/core/dummy.js");
	assertEquals("The content should match", "Success!!", content);

	var empty = mozile.load("nonexistant.js");
	assertNull("There should be no content", empty);
}

/**
 * 
 */
function testFindRoot() {
	assertEquals("The result should be an empty string", "", mozile.findRoot());
}

/**
 * 
 */
function testLoadCore() {
	// If this test suite is working, then loadCore is working.
}

/**
 * 
 */
function testInfo() {
	assertNotNull("Version should exist", mozile.version);
	assertNotNull("Version.major should exist", mozile.version.major);
	assertNotNull("Version.minor should exist", mozile.version.minor);
	assertNotNull("Version.micro should exist", mozile.version.micro);
	assertNotNull("Version.flag should exist", mozile.version.flag);
	var version = mozile.version.major + "." 
		+ mozile.version.minor + "." + 
		mozile.version.micro + mozile.version.flag;
	assertEquals("The versions should match", version, mozile.version.toString());

	assertEquals("About string should exist", "string", typeof(mozile.about) );
	assertEquals("Copyright string should exist", "string", typeof(mozile.copyright) );
	assertEquals("License string should exist", "string", typeof(mozile.license) );
	assertEquals("Homepage string should exist", "string", typeof(mozile.homepage) );
	assertEquals("Credits string should exist", "string", typeof(mozile.credits) );
}

/**
 * 
 */
function testFindModule() {
	// Setup
	mozile.test.core = "string";

	var existing = ["mozile", "mozile.test", "mozile.test.core"];
	for(var i=0; i < existing.length; i++) {
		assertEquals("Should find "+ existing[i], eval(existing[i]), mozile.findModule(existing[i]) );
	}

	var nonexisting = ["foo", "mozile.foo", "mozile.foo.bar"];
	for(var i=0; i < nonexisting.length; i++) {
		assertNull("Should not find "+ nonexisting[i], mozile.findModule(nonexisting[i]) );
	}

	// Teardown
	delete mozile.test.core;
}

/**
 * 
 */
function testLoadModule() {
	// Setup
	var storedRoot = mozile.root;
	mozile.root = mozile.test.root;

	assertEquals("mozile.test.success should not be defined", "undefined", typeof(mozile.test.success));
	assertTrue("Should load tests/core/dummy2.js", mozile.loadModule("mozile.test.core.dummy2") );
	assertEquals("mozile.test.success should be true", true, mozile.test.success);


	var dontLoad = ["mozile", "mozile.test", "foo", "mozile.foo", "mozile.foo.bar"];
	for(var i=0; i < dontLoad.length; i++) {
		assertFalse("Should not load "+ dontLoad[i], mozile.loadModule(dontLoad[i]) );
	}
	
	// Teardown
	mozile.root = storedRoot;
}

/**
 * 
 */
function testRequire() {
	// Currently trivial
}

/**
 * 
 */
function testProvide() {
	// Nothing to test yet.
}

/**
 * 
 */
function testIsPathAbsolute() {
	var path = "http://mozile.mozdev.org";
	assertTrue(path +" is absolute", mozile.isPathAbsolute(path));
	path = "https://mozile.mozdev.org";
	assertTrue(path +" is absolute", mozile.isPathAbsolute(path));
	path = "file:///usr/mozile/";
	assertTrue(path +" is absolute", mozile.isPathAbsolute(path));

	path = "../";
	assertFalse(path +" is not absolute", mozile.isPathAbsolute(path));
	path = "filename";
	assertFalse(path +" is not absolute", mozile.isPathAbsolute(path));
	path = "file?path=http://mozile.mozdev.org";
	assertFalse(path +" is not absolute", mozile.isPathAbsolute(path));
}

/**
 * 
 */
function testGetPath() {
	var url = "http://mozile.mozdev.org";
	var path = "http://mozile.mozdev.org";
	assertEquals("Get path from "+ url, path, mozile.getPath(url));

	var url = "http://mozile.mozdev.org/file.html";
	var path = "http://mozile.mozdev.org/file.html";
	assertEquals("Get path from "+ url, path, mozile.getPath(url));

	var url = "http://mozile.mozdev.org/file.html?cats=dogs";
	var path = "http://mozile.mozdev.org/file.html";
	assertEquals("Get path from "+ url, path, mozile.getPath(url));
}

/**
 * 
 */
function testGetDirectory() {
	var path = "http://mozile.mozdev.org";
	var dir = "http://mozile.mozdev.org";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "http://mozile.mozdev.org/";
	var dir = "http://mozile.mozdev.org/";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "http://mozile.mozdev.org/file.html";
	var dir = "http://mozile.mozdev.org/";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "http://mozile.mozdev.org/file.html?cats=dogs";
	var dir = "http://mozile.mozdev.org/";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "http://mozile.mozdev.org/dir/file.html?cats=dogs";
	var dir = "http://mozile.mozdev.org/dir/";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "file:///usr";
	var dir = "file:///usr";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));

	var path = "file:///usr/local/stuff";
	var dir = "file:///usr/local/";
	assertEquals("Get dir from "+ path, dir, mozile.getDirectory(path));
}

/**
 * 
 */
function testGetAbsolutePath() {
	var loc = document.location.toString();
	var dir = mozile.getDirectory(loc);
	var root = "http://root.com/root/";

	var url = "http://mozile.mozdev.org";
	var path = "http://mozile.mozdev.org";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "http://mozile.mozdev.org/file.html";
	var path = "http://mozile.mozdev.org/file.html";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "http://mozile.mozdev.org/file.html?cats=dogs";
	var path = "http://mozile.mozdev.org/file.html?cats=dogs";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "file.html";
	var path = dir +"file.html";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "../file.html";
	var path = dir +"../file.html";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "file.html?cats=dogs";
	var path = dir +"file.html?cats=dogs";
	assertEquals("Get path from "+ url, path, mozile.getAbsolutePath(url));

	var url = "file.html";
	var path = root +"file.html";
	assertEquals("Get path from "+ url +" and "+ root, path, mozile.getAbsolutePath(url, root));

	var url = "../file.html";
	var path = root +"../file.html";
	assertEquals("Get path from "+ url +" and "+ root, path, mozile.getAbsolutePath(url, root));

	var url = "file.html?cats=dogs";
	var path = root +"file.html?cats=dogs";
	assertEquals("Get path from "+ url +" and "+ root, path, mozile.getAbsolutePath(url, root));
}

/**
 * 
 */
function testAlert() {
	// Currently trivial
}

/**
 * 
 */
function testDumpError() {
	// Currently trivial
}

/**
 * 
 */
function testEditAllText() {
	// Test basic
	mozile.editAllText(false);
	assertEquals("Check navigateLeftRight", mozile.edit.navigateLeftRight, mozile.edit._defaultCommands[0]);
	assertEquals("Check insertText", mozile.edit.insertText, mozile.edit._defaultCommands[1]);
	assertEquals("Check removeText", mozile.edit.removeText, mozile.edit._defaultCommands[2]);
	mozile.edit._defaultCommands = null;

	// Test rich
	mozile.editAllText(true);
	assertEquals("Check navigateLeftRight", mozile.edit.navigateLeftRight, mozile.edit._defaultCommands[0]);
	assertEquals("Check insertText", mozile.edit.insertText, mozile.edit._defaultCommands[1]);
	assertEquals("Check removeText", mozile.edit.removeText, mozile.edit._defaultCommands[2]);
	assertEquals("Check remove", mozile.edit.remove, mozile.edit._defaultCommands[3]);
	assertEquals("Check splitBlocks", "splitBlocks", mozile.edit._defaultCommands[4].name);
	mozile.edit._defaultCommands = null;
	
	
}

/**
 * 
 */
function testEditElement() {
	mozile.require("mozile.dom");
	mozile.require("mozile.edit");
	var target = document.getElementById("target");
	
	// Use an id
	var i = 1;
	mozile.editElement("target");
	assertNotUndefined(i +" There should now be a mozile property.", target.mozile);
	assertTrue(i +" The target should be editable.", mozile.edit.getMark(target, "editable"));
	assertUndefined(i +" contentEditable should be empty.", mozile.edit.getMark(target, "contentEditable"));
	target.mozile = null;
	
	// Use a node
	i = 2;
	mozile.editElement(target);
	assertNotUndefined(i +" There should now be a mozile property.", target.mozile);
	assertTrue(i +" The target should be editable.", mozile.edit.getMark(target, "editable"));
	// TODO: Fix this test in IE.
	//assertUndefined(i +" contentEditable should be empty.", mozile.edit.getMark(target, "contentEditable"));
	target.mozile = null;

	// Use a node
	i = 3;
	var container = document.getElementById("container");
	mozile.editElement("container");
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.",  mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;
}

/**
 * 
 */
function testEditElements() {
	mozile.require("mozile.dom");
	mozile.require("mozile.edit");
	var container = document.getElementById("container");

	// Use a class
	var i = 1;
	mozile.editElements("editor");
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.", mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;

	// Use an array
	i = 2;
	mozile.editElements([container]);
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.", mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;

	// Use a node list
	i = 3;
	var list = document.getElementsByTagName("div");
	mozile.editElements(list);
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.", mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;

	// Use an attribute
	i = 4;
	mozile.editElements("true", "contentEditable");
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.", mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;

	// Use an element name
	i = 5;
	mozile.editElements("div", null);
	assertNotUndefined(i +" There should now be a mozile property.", container.mozile);
	assertTrue(i +" The container should be editable.", mozile.edit.getMark(container, "editable"));
	assertTrue(i +" contentEditable should be 'true'.", mozile.edit.getMark(container, "contentEditable"));
	container.mozile = null;
}

/**
 * 
 */
function testEditDocument() {
	mozile.require("mozile.dom");
	mozile.require("mozile.edit");
	var target = document.documentElement;
	
	mozile.editDocument();
	assertNotUndefined("There should now be a mozile property.", target.mozile);
	assertTrue("The target should be editable.", mozile.edit.getMark(target, "editable"));
	//assertUndefined("contentEditable should be empty.", mozile.edit.getMark(target, "contentEditable"));
	target.mozile = null;}

/**
 * 
 */
function testUseSchema() {
	assertEquals("There should be no schema", "undefined", typeof(mozile.schema));
	assertTrue("The schema should be successfully loaded", mozile.useSchema(testSchema));
	assertNotUndefined("There should be a schema", mozile.schema);
}




