/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Contributor(s):
 *   C.N Medappa <jrex_moz@yahoo.com><>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the NPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the NPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package org.mozilla.jrex.ui;

import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import java.util.Vector;
import java.util.Iterator;

import org.mozilla.jrex.exception.JRexException;
import org.mozilla.jrex.log.JRexL;
import org.mozilla.jrex.navigation.*;
import org.mozilla.jrex.event.history.*;
import org.mozilla.jrex.event.uricontent.*;
import org.mozilla.jrex.event.progress.*;
import org.mozilla.jrex.event.context.*;
import org.mozilla.jrex.event.tooltip.*;


/**
* JRexCanvas is the browser component. This component wraps the <b>nsWebBrowser</b>
* and related interfaces of mozilla embedding.
* It allows listening to follwing events:
* <ul>
* <li><b>ProgressListener</b></li>
* <li><b>ContextMenuListener</b></li>
* <li><b>HistoryListener</b></li>
* <li><b>KeyListener</b></li>
* <li><b>MouseListener</b></li>
* <li><b>MouseMotionListener</b></li>
* <li><b>URIContentListener</b></li>
* </ul>
*
* The Mouse and Key events are posted directly into Java's SystemEvent queue. So there is
* no specific Mouse and Key Listener for JRexCanvas.
* All the event except URIContentListener are mulitcast events. i.e only one URIContentListener
* can be added to a browser component.
* Listening to ToolTip events not enabled at present. It is used only for internal use.
* It allows operation related to Navigation, Edit and Print actions.
*
* @author <a href="mailto:jrex_moz@yahoo.com">C.N.Medappa</a>
* @version 1.0
* see org.mozilla.jrex.navigation.WebNavigation
*/

public class JRexCanvas extends java.awt.Canvas {
	private static final String className	= JRexCanvas.class.getName();

	/**
	* PrintPreview Navigation Constants
	*/
	private static final int PRINTPREVIEW_GOTO_PAGENUM	= 0;
	private static final int PRINTPREVIEW_PREV_PAGE 	= 1;
	private static final int PRINTPREVIEW_NEXT_PAGE		= 2;
	private static final int PRINTPREVIEW_HOME 			= 3;
	private static final int PRINTPREVIEW_END 			= 4;

	/**
	* Indicates whether the native initIDs was successfully.
	*/
	//TO DO this should go
	private static boolean inited			= false;

	static{
		try{
			initIDs();
			inited=true;
		}catch(JRexException ex){
			JRexL.printStackTrace(ex);
			throw new RuntimeException(ex);
		}
	}

	/**
	* Represents the peer browser component associated with browser component.
	*/
	private int			jrexPeer			= 0;

	/**
	* Indicates whether the peer browser component is realized.
	*/
	private boolean 	realized			= false;

	/**
	* Indicates whether the peer has been notified of the java component peer creation.
	* This is required for *nix, since the java.awt.Component's addnotify function actualy does not create
	* a peer component in *nix(at least in Linux).
	*/
	private boolean 	notifyPeer			= false;

	/**
	* Indicates whether ProgressListener is added for the browser component.
	*/
	private boolean 	progLstnAdded		= false;

	/**
	* Indicates whether ContextMenuListener is added for the browser component.
	*/
	private boolean 	ctxMenuLstnAdded	= false;

	/**
	* Indicates whether HistoryListener is added for the browser component.
	*/
	private boolean 	hisLstnAdded		= false;

	/**
	* Indicates whether KeyListener is added for the browser component.
	*/
	private boolean 	keyLstnAdded		= false;

	/**
	* Indicates whether MouseListener is added for the browser component.
	*/
	private boolean 	mouseLstnAdded		= false;

	/**
	* Indicates whether MouseMotionListener is added for the browser component.
	*/
	private boolean 	mouseMotionLstnAdded= false;

	/**
	* Indicates whether URIContentListener is added for the browser component.
	*/
	private boolean 	uriContentLstnAdded = false;

	/**
	* Indicates whether the component should be realized after creation.
	*/
	private boolean 	doRrealize			= false;

	/**
	* Variable holding the unique-id associated with the browser component creation request.
	*/
	private int		 	waitKey				= 0;

	/**
	* Variable holding the unique-id associated with the browser component creation request.
	*/
	private WebNavigation navigator			= new WebNavigationImpl();

	/**
	* A list of ProgressListener added to browser.
	*/
	private Vector progLsnList;

	/**
	* A list of ContextMenuListener added to browser.
	*/
	private Vector ctxMenuLsnList;

	/**
	* A list of HistoryListener added to browser.
	*/
	private Vector hisLsnList;

	/**
	* URIContentListener added to browser.
	*/
	private URIContentListener uriContentListener;

	/**
	* Constructor to create JRexCanvas.
	* @param	doRrealize <code>true</code> indicates that peer will be realized from java.
	* @param	waitKey The unique-id associated with JRexCanvas creation.
				this is used by WindowManager.
	*/
	protected JRexCanvas(boolean doRrealize, int waitKey){
		this.doRrealize=doRrealize;
		this.waitKey=waitKey;
		setFocusable(true);
		enableEvents(AWTEvent.FOCUS_EVENT_MASK);
	}


	/**
	* Use this function to create a JRexCanvas.
	* @param	doRrealize <code>true</code> to realize peer from java.
	*/
	public static JRexCanvas createBrowserComponent(boolean doRrealize){
		return createBrowserComponent(doRrealize, 0);
	}

	/**
	* Use this function to create a JRexCanvas. This function is meant for
	* WindowManger to handle WindowCreation request from embedding engine.
	* @param	doRrealize <code>true</code> to realize peer from java.
				Note:- doRrealize should be <code>false</code> for WindowCreation
				request from embedding engine to avoid deadlock(Perticluarly when dealing with modal dialogs).
	* @param	waitKey The unique-id associated with JRexCanvas creation.
				this is used by WindowManager.
	*/
	public static JRexCanvas createBrowserComponent(boolean doRrealize, int waitKey){
		if(inited)
			return new JRexCanvas(doRrealize, waitKey);
		throw new java.awt.IllegalComponentStateException("Invocation of initIDs Failed!!!");
	}

	/**
	* Return peer component associated with JRexCanvas.
	*/
	public int getJRexPeer(){
		return jrexPeer;
	}

    /**
     * Creates the peer of the JRexCanvas. If doRrealize is set to <code>true</code> the native component
     * is realized and made visible.
     * This function is also used to associate the JRex peer with java canvas peer in windows platform.
     * @see     java.awt.Canvas#addNotify()
     */
	public void addNotify(){
		super.addNotify();
		if(JRexL.on)JRexL.log(className,"**** addNotify() -->jrexPeer<"+jrexPeer+"> before CreatePeer ****");
		if(jrexPeer>0)return;
		synchronized (getTreeLock()) {
			try{
				if(JRexL.on)JRexL.log(className,"**** addNotify() -->addNotify currentThread <"+Thread.currentThread()+">****");
				CreatePeer(waitKey);
				if(JRexL.on)JRexL.log(className,"**** addNotify() -->jrexPeer<"+jrexPeer+"> doRrealize<"+doRrealize+"> after CreatePeer **** ");
				if(doRrealize){
					RealizePeer();
					browserRealized();
					if(JRexL.on)JRexL.log(className,"**** addNotify() -->RealizePeer done **** ");
					setVisible(super.isVisible());
				}
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
				throw new RuntimeException("**** addNotify() --> createPeer FAILED <"+ex.getMessage()+">****");
			}
		}
	}

	/**
	* This function is used to discard a window creation request from embedding engine.
	* This will notify the native counterpart to abort window creation request identified by id <code>waitKey</code>
	* This function is intended to be used by WindowManager.
	*/
	public static void abortBrowserCreation(int waitKey){
		try{
			AbortPeerCreation(waitKey);
		}catch(JRexException ex){
			JRexL.printStackTrace(ex);
		}
		return;
	}

	/**
	* Invoked to indicate that browser peer has completed realization.
	* This function is intended to be used by WindowManager.
	*/
	//don't call form outside
	public void browserRealized(){
		if(JRexL.on)JRexL.log(className,"**** browserRealized() -->realized <"+realized+"> **** ");
		if(realized)return;
		realized=true;
		if(progLsnList!=null && progLsnList.size()>0)
			addProgressListenerInternal();
		if(ctxMenuLsnList!=null && ctxMenuLsnList.size()>0)
			addContextMenuListenerInternal();
		if(hisLsnList!=null && hisLsnList.size()>0)
			addHistoryListenerInternal();
		if(uriContentListener!=null)
			addURIContentListenerInternal();
		if(getKeyListeners().length>0)
			addKeyListenerInternal();
		if(getMouseListeners().length>0)
			addMouseListenerInternal();
		if(getMouseMotionListeners().length>0)
			addMouseMotionListenerInternal();
	}

    /**
     * Processes focus events occurring on this component by dispatching them to native peer.
     * @see     java.awt.Component#processFocusEvent(FocusEvent)
     */
	protected void processFocusEvent(final FocusEvent fe){
		if(JRexL.on)JRexL.log(className,"processFocusEvent()-->***** I AM HERE fe<"+fe+">***** ");
		if(fe.isTemporary() || !realized || !isShowing())return;
		if(JRexL.on)JRexL.log(className,"processFocusEvent()-->***** FOCUS_GAINED<"+(fe.getID()==FocusEvent.FOCUS_GAINED)+"> ***** ");

		if(EventQueue.isDispatchThread()){
			try{
				SetFocusPeer(fe.getID()==FocusEvent.FOCUS_GAINED);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
			return;
		}
		EventQueue.invokeLater(new Runnable(){
		  public void run(){
			try{
				SetFocusPeer(fe.getID()==FocusEvent.FOCUS_GAINED);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}});
	}

    /**
     * Paints the JRexCanvas by signalling Repaint to the peer component.
     * This function is also used to associate the JRex peer with java canvas peer in
     * *nix platform. since the java.awt.Component's addnotify function actualy does not create
	 * a java peer component in addnotify call.
     *
     * @param      g   the specified Graphics context
     * @see        java.awt.Component#paint(Graphics)
     */
	public void paint(java.awt.Graphics g){
		if(JRexL.on)JRexL.log(className,"paint()-->***** I AM HERE ***** ");
		if(jrexPeer>0 && !notifyPeer){
			notifyPeer=true;
			try{
				if(JRexL.on)JRexL.log(className,"paint()-->***** Calling GotVisual ***** ");
				synchronized (getTreeLock()) {
					GotVisual();
				}
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
		if(!realized || !isShowing())return;
		try{
			if(JRexL.on)JRexL.log(className,"paint()-->***** Calling RepaintPeer ***** ");
			RepaintPeer(false);
		}catch(JRexException ex){
			JRexL.printStackTrace(ex);
		}
	}

	/**
	* Simply calls paint function.
	* @param      g   the specified Graphics context
	* @see        #paint(Graphics)
	*/
	public void update(java.awt.Graphics g){
		paint(g);
	}

    /**
     * Destroys the peer of the JRexCanvas.
     * @see     java.awt.Canvas#removeNotify()
     */
	public void removeNotify(){
		super.removeNotify();
		if(JRexL.on)JRexL.log(className,"removeNotify()--> ***** jrexPeer<"+jrexPeer+"> before DestroyPeer ****");
		try{
			DestroyPeer();
			if(JRexL.on)JRexL.log(className,"removeNotify()-->***** jrexPeer <"+jrexPeer+"> after DestroyPeer ****");
		}catch(JRexException ex){
			JRexL.printStackTrace(ex);
		}finally{
			jrexPeer=0;
		}
	}


	/**
	* Sets the visible state of the JRexCanvas.
	* @param visible if true, shows this JRexCanvas peer; otherwise, hides it
	*/
	public void setVisible(final boolean visible){
		super.setVisible(visible);
		if(JRexL.on)JRexL.log(className,"setVisible()-->***** visible <"+visible+"> ****");
		if(!realized)return;
		if(EventQueue.isDispatchThread()){
			try{
				SetVisiblePeer(visible);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
			return;
		}
		EventQueue.invokeLater(new Runnable(){
		  public void run(){
			try{
				SetVisiblePeer(visible);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}});
	}


	/**
	* Determines whether this component should be visible when its
	* parent is visible.
	* @see #setVisible
	*/
	public boolean	isVisible(){
		if(JRexL.on)JRexL.log(className,"isVisible()-->**** I AM HERE ****");
		if(realized){
			try{
				boolean temp= IsVisiblePeer();
				if(JRexL.on)JRexL.log(className,"isPeerVisible()-->***** visible <"+temp+">***** ");
				return temp;
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
		return super.isVisible();
	}

	/**
	* Enables or disables this component, depending on the value of the
	* parameter <code>enable</code>.
	*/
    public void	setEnabled(final boolean enable){
		if(JRexL.on)JRexL.log(className,"setEnabled()-->***** enable <"+enable+"> ***** ");
		super.setEnabled(enable);
		if(!realized)return;
		if(EventQueue.isDispatchThread()){
			try{
				SetEnabledPeer(enable);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
			return;
		}
		EventQueue.invokeLater(new Runnable(){
		  public void run(){
			try{
				SetEnabledPeer(enable);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}});
	}

	/**
	* Determines whether this component is enabled.
	* @see #setEnabled
	*/
    public boolean isEnabled(){
		if(JRexL.on)JRexL.log(className,"isEnabled()-->**** I AM HERE ****");
		if(realized){
			try{
				boolean temp=IsEnabledPeer();
				if(JRexL.on)JRexL.log(className,"IsEnabledPeer()-->**** enable <"+temp+">****");
				return temp;
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
		return super.isEnabled();
	}

	/**
	* Moves and resizes this component. The new location of the top-left
	* corner is specified by <code>x</code> and <code>y</code>, and the
	* new size is specified by <code>width</code> and <code>height</code>.
	* @param x the new <i>x</i>-coordinate of this component
	* @param y the new <i>y</i>-coordinate of this component
	* @param width the new <code>width</code> of this component
	* @param height the new <code>height</code> of this
	* 		component
	* @see #getBounds
	*/
	public void setBounds(int x, int y, int width, int height){
		if(JRexL.on)JRexL.log(className,"setBounds()-->**** x<"+x+"> y<"+y+"> width<"+width+"> height<"+height+"> ***");
		super.setBounds(x,y,width,height);
		if(!realized)return;
		if(EventQueue.isDispatchThread()){
			try{
				SetBoundsPeer();
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
			return;
		}

		EventQueue.invokeLater(new Runnable(){
		  public void run(){
			try{
				SetBoundsPeer();
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}});
	}

	/**
	* Gets the bounds of this component in the form of a
	* <code>Rectangle</code> object. The bounds specify this
	* component's width, height, and location relative to
	* its parent.
	* @return a rectangle indicating this component's bounds
	* @see #setBounds
	*/
	public Rectangle getBounds(){
		if(JRexL.on)JRexL.log(className,"getBounds()-->**** I AM HERE ***");
		if(realized){
			try{
				return GetBoundsPeer();
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
		return super.getBounds();
	}

	/**
	* Gets the mininimum size of this component.
	* @see #getSize
	* @see java.awt.LayoutManager
	*/
	public Dimension getMinimumSize(){
		if(JRexL.on)JRexL.log(className,"getMinimumSize()-->**** I AM HERE ***");
		return getSize();
	}

	/**
	* Returns the size of this object in the form of a
	* <code>Dimension</code> object. The height field of the
	* <code>Dimension</code> object contains this objects's
	* height, and the width field of the <code>Dimension</code>
	* object contains this object's width.
	*
	* @return a <code>Dimension</code> object that indicates
	*     the size of this component; <code>null</code> if
	*     this object is not on the screen
	*/
	public Dimension getSize(){
		if(JRexL.on)JRexL.log(className,"getSize()-->**** I AM HERE ***");
		return realized?getBounds().getSize():super.getSize();
	}

	/**
	* Gets the preferred size of this component.
	* @return a dimension object indicating this component's preferred size
	* @see #getSize
	*/
    public Dimension getPreferredSize() {
		if(JRexL.on)JRexL.log(className,"getPreferredSize()-->**** I AM HERE ***");
		return getSize();
    }


	/**
	* Gets the JRex Window Title.
	* @return the tittle associated with the browser component
	* @see #setJRexWindowTitle
	*/
	public String getJRexWindowTitle(){
		if(realized){
			try{
				return GetTitlePeer();
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
		return "";
	}

	/**
	* Sets the JRex Window Title.
	* @param	title the tittle to be associated with the browser component
	* @see #getJRexWindowTitle
	*/
	public void setJRexWindowTitle(String title){
		if(realized){
			try{
				SetTitlePeer(title);
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}


	/**
	* Returns the navigator assocaiated with the browser component.
	* see org.mozilla.jrex.navigation.WebNavigation
	*/
	public WebNavigation getNavigator(){
		navigator.setJRexPeer(jrexPeer);
		return navigator;
	}

	public void addKeyListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addKeyListenerInternal() -->keyLstnAdded <"+keyLstnAdded+"> **** ");
		if(keyLstnAdded)return;
		if(realized){
			try{
				AddKeyListener();
				keyLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addKeyListenerInternal() -->keyLstnAdded <"+keyLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Adds the specified key listener to receive key events from
     * this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    l   the key listener.
     * @see      java.awt.event.KeyEvent
     * @see      java.awt.event.KeyListener
     * @see      #removeKeyListener
     * @see      java.awt.Component#getKeyListeners
     */
	public void addKeyListener(KeyListener l){
		if(JRexL.on)JRexL.log(className,"**** addKeyListener() -->l <"+l+"> **** ");
		if(l==null)return;
		super.addKeyListener(l);
		addKeyListenerInternal();

	}

	public void addMouseListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addMouseListenerInternal() -->mouseLstnAdded <"+mouseLstnAdded+"> **** ");
		if(mouseLstnAdded)return;
		if(realized){
			try{
				AddMouseListener();
				mouseLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addMouseListenerInternal() -->mouseLstnAdded <"+mouseLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Adds the specified mouse listener to receive mouse events from
     * this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    l   the mouse listener
     * @see      java.awt.event.MouseEvent
     * @see      java.awt.event.MouseListener
     * @see      #removeMouseListener
     * @see      java.awt.Component#getMouseListeners
     */
	public void addMouseListener(MouseListener l){
		if(JRexL.on)JRexL.log(className,"**** addMouseListener() -->l <"+l+"> **** ");
		if(l==null)return;
		super.addMouseListener(l);
		addMouseListenerInternal();
	}


	public void addMouseMotionListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addMouseMotionListenerInternal() -->mouseMotionLstnAdded <"+mouseMotionLstnAdded+"> **** ");
		if(mouseMotionLstnAdded)return;
		if(realized){
			try{
				AddMouseMotionListener();
				mouseMotionLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addMouseMotionListenerInternal() -->mouseMotionLstnAdded <"+mouseMotionLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Adds the specified mouse motion listener to receive mouse motion
     * events from this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    l   the mouse motion listener
     * @see      java.awt.event.MouseEvent
     * @see      java.awt.event.MouseMotionListener
     * @see      #removeMouseMotionListener
     * @see      java.awt.Component#getMouseMotionListeners
     */
	public void addMouseMotionListener(MouseMotionListener l){
		if(JRexL.on)JRexL.log(className,"**** addMouseMotionListener() -->l <"+l+"> **** ");
		if(l==null)return;
		super.addMouseMotionListener(l);
		addMouseMotionListenerInternal();
	}


	private void addProgressListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addProgressListenerInternal() -->progLstnAdded <"+progLstnAdded+"> **** ");
		if(progLstnAdded)return;
		if(realized){
			try{
				AddProgressListener();
				progLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addProgressListenerInternal() -->progLstnAdded <"+progLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Adds the specified progress listener to receive progress
     * events from this component.
     * If listener <code>progressListener</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    progressListener the Progress listener
     * @see      org.mozilla.jrex.event.progress.ProgressEvent
     * @see      #removeProgressListener
     * @see      #getProgressListeners
     */
	public void addProgressListener(ProgressListener progressListener){
		if(JRexL.on)JRexL.log(className,"**** addProgressListener() -->progressListener <"+progressListener+"> **** ");
		if(progressListener==null)return;
		if(progLsnList==null)progLsnList=new Vector();
		if(!progLsnList.contains(progressListener))
			progLsnList.add(progressListener);
		addProgressListenerInternal();
	}


	/**
	* Returns an array of all the progress listeners
	* registered on this component.
	*
	* @return all of this component's <code>ProgressListener</code>s
	*         or an empty array if no progress
	*         listeners are currently registered
	*
	* @see      #addProgressListener
	* @see      #removeProgressListener
	*/
    public synchronized ProgressListener[] getProgressListeners() {
		if(progLsnList==null)return new ProgressListener[0];
		ProgressListener[] list= new ProgressListener[progLsnList.size()];
        return (ProgressListener[]) progLsnList.toArray(list);
    }

	private void addContextMenuListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addContextMenuListenerInternal() -->ctxMenuLstnAdded <"+ctxMenuLstnAdded+"> **** ");
		if(ctxMenuLstnAdded)return;
		if(realized){
			try{
				AddContextMenuListener();
				ctxMenuLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addContextMenuListenerInternal() -->ctxMenuLstnAdded <"+ctxMenuLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Adds the specified ContextMenu listener to receive ContextMenu
     * events from this component.
     * If listener <code>contextMenuListener</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    contextMenuListener   the ContextMenu listener
     * @see      org.mozilla.jrex.event.context.ContextMenuEvent
     * @see      #removeContextMenuListener
     * @see      #getContextMenuListeners
     */

	public void addContextMenuListener(ContextMenuListener contextMenuListener){
		if(JRexL.on)JRexL.log(className,"**** addContextMenuListener() -->contextMenuListener <"+contextMenuListener+"> **** ");
		if(contextMenuListener==null)return;
		if(ctxMenuLsnList==null)ctxMenuLsnList=new Vector();
		if(!ctxMenuLsnList.contains(contextMenuListener))
			ctxMenuLsnList.add(contextMenuListener);
		addContextMenuListenerInternal();
	}


	/**
	* Returns an array of all the ContextMenu listeners
	* registered on this component.
	*
	* @return all of this component's <code>ContextMenuListener</code>s
	*         or an empty array if no ContextMenu
	*         listeners are currently registered
	*
	* @see      #addContextMenuListener
	* @see      #removeContextMenuListener
	*/
    public synchronized ContextMenuListener[] getContextMenuListeners() {
		if(ctxMenuLsnList==null)return new ContextMenuListener[0];
		ContextMenuListener[] list= new ContextMenuListener[ctxMenuLsnList.size()];
        return (ContextMenuListener[]) ctxMenuLsnList.toArray(list);
    }



	private void addHistoryListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addHistoryListenerInternal() -->hisLstnAdded <"+hisLstnAdded+"> **** ");
		if(hisLstnAdded)return;
		if(realized){
			try{
				AddHistoryListener();
				hisLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addHistoryListenerInternal() -->hisLstnAdded <"+hisLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}


    /**
     * Adds the specified History listener to receive History
     * events from this component.
     * If listener <code>historyListener</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * @param    historyListener   the History listener
     * @see      org.mozilla.jrex.event.history.HistoryEvent
     * @see      #removeHistoryListener
     * @see      #getHistoryListeners
     */

	public void addHistoryListener(HistoryListener historyListener){
		if(JRexL.on)JRexL.log(className,"**** addHistoryListener() -->historyListener <"+historyListener+"> **** ");
		if(historyListener==null)return;
		if(hisLsnList==null)hisLsnList=new Vector();
		if(!hisLsnList.contains(historyListener))
			hisLsnList.add(historyListener);
		addHistoryListenerInternal();
	}


	/**
	* Returns an array of all the History listeners
	* registered on this component.
	*
	* @return all of this component's <code>HistoryListener</code>s
	*         or an empty array if no History
	*         listeners are currently registered
	*
	* @see      #addHistoryListener
	* @see      #removeHistoryListener
	*/
    public synchronized HistoryListener[] getHistoryListeners() {
		if(hisLsnList==null)return new HistoryListener[0];
		HistoryListener[] list= new HistoryListener[hisLsnList.size()];
        return (HistoryListener[]) ctxMenuLsnList.toArray(list);
    }


	private void addURIContentListenerInternal(){
		if(JRexL.on)JRexL.log(className,"**** addURIContentListener() -->uriContentLstnAdded <"+uriContentLstnAdded+"> **** ");
		if(uriContentLstnAdded)return;
		if(realized){
			try{
				AddURIContentListener();
				uriContentLstnAdded=true;
				if(JRexL.on)JRexL.log(className,"**** addURIContentListener() -->uriContentLstnAdded <"+uriContentLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

	/**
	* Adds the specified URIContent listener to receive URIContent
	* events from this component.
	* If listener <code>uriContentListener</code> is <code>null</code>,
	* no exception is thrown and no action is performed.
	* If listener was already added the <code>uriContentListener</code> will relace it.
	*
	* @param    uriContentListener   the URIContent listener
	* @see      org.mozilla.jrex.event.uricontent.URIContentEvent
	* @see      #removeURIContentListener
	*/
	public void addURIContentListener(URIContentListener uriContentListener){
		if(JRexL.on)JRexL.log(className,"**** addURIContentListener() -->uriContentListener <"+uriContentListener+"> **** ");
		if(uriContentListener==null)return;
		this.uriContentListener=uriContentListener;
		addURIContentListenerInternal();
	}


    /**
     * Removes the specified key listener so that it no longer
     * receives key events from this component. This method performs
     * no function, nor does it throw an exception, if the listener
     * specified by the argument was not previously added to this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * If list of KeyListeners is empty, native peer is requested to remove the KeyListener.
     *
     * @param    l   the key listener
     * @see      java.awt.event.KeyEvent
     * @see      java.awt.event.KeyListener
     * @see      #addKeyListener
     * @see      #getKeyListeners
     */

	public void removeKeyListener(KeyListener l){
		super.removeKeyListener(l);
		if(getKeyListeners().length==0 && keyLstnAdded){
			try{
				RemoveKeyListener();
				keyLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeKeyListener() -->mouseLstnAdded <"+mouseLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

    /**
     * Removes the specified mouse listener so that it no longer
     * receives mouse events from this component. This method performs
     * no function, nor does it throw an exception, if the listener
     * specified by the argument was not previously added to this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * If list of MouseListener is empty, native peer is requested to remove the MouseListener.
     *
     * @param    l   the mouse listener
     * @see      java.awt.event.MouseEvent
     * @see      java.awt.event.MouseListener
     * @see      #addMouseListener
     * @see      #getMouseListeners
     */
	public void removeMouseListener(MouseListener l){
		super.removeMouseListener(l);
		if(getMouseListeners().length==0 && realized){
			try{
				RemoveMouseListener();
				mouseLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeMouseListener() -->mouseLstnAdded <"+mouseLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}

	}


    /**
     * Removes the specified mouse motion listener so that it no longer
     * receives mouse motion events from this component. This method performs
     * no function, nor does it throw an exception, if the listener
     * specified by the argument was not previously added to this component.
     * If listener <code>l</code> is <code>null</code>,
     * no exception is thrown and no action is performed.
     *
     * If list of MouseMotionListener is empty, native peer is requested to remove the MouseMotionListener.
     *
     * @param    l   the mouse motion listener
     * @see      java.awt.event.MouseEvent
     * @see      java.awt.event.MouseMotionListener
     * @see      #addMouseMotionListener
     * @see      #getMouseMotionListeners
     */
	public void removeMouseMotionListener(MouseMotionListener l){
		super.removeMouseMotionListener(l);
		if(getMouseMotionListeners().length==0 && mouseMotionLstnAdded){
			try{
				RemoveMouseMotionListener();
				mouseMotionLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeMouseMotionListener() -->mouseMotionLstnAdded <"+mouseMotionLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

	/**
	* Removes the specified progress listener so that it no longer
	* receives progress events from this component. This method performs
	* no function, nor does it throw an exception, if the listener
	* specified by the argument was not previously added to this component.
	* If listener <code>progressListener</code> is <code>null</code>,
	* no exception is thrown and no action is performed.
	*
	* If list of ProgressListener is empty, native peer is requested to remove the ProgressListener.
	*
	* @param    progressListener   the progress listener
	* @see      org.mozilla.jrex.event.progress.ProgressEvent
	* @see      #addProgressListener
	* @see      #getProgressListeners
	*/
	public void removeProgressListener(ProgressListener progressListener){
		if(JRexL.on)JRexL.log(className,"**** removeProgressListener() -->progressListener <"+progressListener+"> **** ");
		if(progressListener==null|| progLsnList==null)return;
		progLsnList.remove(progressListener);
		if(progLsnList.size()<1 && progLstnAdded){
			try{
				RemoveProgressListener();
				progLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeProgressListener() -->progLstnAdded <"+progLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}

	/**
	* Removes the specified ContextMenu listener so that it no longer
	* receives ContextMenu events from this component. This method performs
	* no function, nor does it throw an exception, if the listener
	* specified by the argument was not previously added to this component.
	* If listener <code>contextMenuListener</code> is <code>null</code>,
	* no exception is thrown and no action is performed.
	*
	* If list of ContextMenuListener is empty, native peer is requested to remove the ContextMenuListener.
	*
	* @param    contextMenuListener   the ContextMenu listener
	* @see      org.mozilla.jrex.event.context.ContextMenuEvent
	* @see      #addContextMenuListener
	* @see      #getContextMenuListeners
	*/
	public void removeContextMenuListener(ContextMenuListener contextMenuListener){
		if(JRexL.on)JRexL.log(className,"**** removeContextMenuListener() -->contextMenuListener <"+contextMenuListener+"> **** ");
		if(contextMenuListener==null||ctxMenuLsnList==null)return;
		ctxMenuLsnList.remove(contextMenuListener);
		if(ctxMenuLsnList.size()<1 && ctxMenuLstnAdded){
			try{
				RemoveContextMenuListener();
				ctxMenuLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeContextMenuListener() -->ctxMenuLstnAdded <"+ctxMenuLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}


	/**
	* Removes the specified History listener so that it no longer
	* receives History events from this component. This method performs
	* no function, nor does it throw an exception, if the listener
	* specified by the argument was not previously added to this component.
	* If listener <code>historyListener</code> is <code>null</code>,
	* no exception is thrown and no action is performed.
	*
	* If list of HistoryListener is empty, native peer is requested to remove the HistoryListener.
	*
	* @param    historyListener   the History listener
	* @see      org.mozilla.jrex.event.history.HistoryEvent
	* @see      #addHistoryListener
	* @see      #getHistoryListeners
	*/
	public void removeHistoryListener(HistoryListener historyListener){
		if(JRexL.on)JRexL.log(className,"**** removeHistoryListener() -->historyListener <"+historyListener+"> **** ");
		if(historyListener==null || hisLsnList==null)return;
		hisLsnList.remove(historyListener);
		if(hisLsnList.size()<1 && hisLstnAdded){
			try{
				RemoveHistoryListener();
				hisLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeHistoryListener() -->hisLstnAdded <"+hisLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}


	/**
	* Removes the URIContent listener so that it no longer
	* receives URIContent events from this component. This method performs
	* no function, nor does it throw an exception, if the listener was
	* not previously added to this component.
	*
	* Native peer is requested to remove the URIContentListener.
	*
	* @see      org.mozilla.jrex.event.uricontent.URIContentEvent
	* @see      #addURIContentListener
	*/
	public void removeURIContentListener(){
		if(JRexL.on)JRexL.log(className,"**** removeURIContentListener() -->uriContentListener <"+uriContentListener+"> **** ");
		if(uriContentLstnAdded){
			try{
				RemoveURIContentListener();
				uriContentLstnAdded=false;
				if(JRexL.on)JRexL.log(className,"**** removeURIContentListener() -->uriContentLstnAdded <"+uriContentLstnAdded+"> **** ");
			}catch(JRexException ex){
				JRexL.printStackTrace(ex);
			}
		}
	}


   /**
     * Processes progress events occurring on this browser
     * by dispatching them to any registered
     * <code>ProgressListener</code> objects.
     * <p>
     * This method is not called unless progress events are
     * enabled for this browser. Progress events are enabled
     * when an <code>ProgressListener</code> object is registered
     * via <code>addProgressListener</code>.
     *
     * @param       pe the Progress event
     * @see         org.mozilla.jrex.event.progress.ProgressListener
     */
    public void processProgressEvent(ProgressEvent pe) {
		if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->pe<"+pe+"> ****");
		assert progLsnList!=null && progLsnList.size()>0;
		Iterator progIterator= progLsnList.iterator();
		int progEventType=pe.getEventType();
		switch(progEventType){
			case ProgressEventConstants.PROG_STATE_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onStateChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onStateChange ****");
				break;

			case ProgressEventConstants.PROG_PROGRESS_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onProgressChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onProgressChange ****");
				break;

			case ProgressEventConstants.PROG_LOC_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onLocationChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onLocationChange ****");
				break;

			case ProgressEventConstants.PROG_STATUS_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onStatusChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onStatusChange ****");
				break;

			case ProgressEventConstants.PROG_LINK_STATUS_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onLinkStatusChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onStatusChange ****");
				break;

			case ProgressEventConstants.PROG_SECURITY_CHANGE:
				while(progIterator.hasNext())
					((ProgressListener)progIterator.next()).onSecurityChange(pe);
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()-->onSecurityChange ****");
				break;
			default:
				if(JRexL.on)JRexL.log(className,"**** processProgressEvent()--> UNKNOW TYPE ****");
		};
    }


   /**
     * Processes ContextMenu events occurring on this browser
     * by dispatching them to any registered
     * <code>ContextMenuListener</code> objects.
     * <p>
     * This method is not called unless ContextMenu events are
     * enabled for this browser. ContextMenu events are enabled
     * when an <code>ContextMenuListener</code> object is registered
     * via <code>addContextMenuListener</code>.
     *
     * @param       ce the ContextMenu event
     * @see         org.mozilla.jrex.event.context.ContextMenuListener
     */
    public void processContextMenuEvent(ContextMenuEvent ce){
		if(JRexL.on)JRexL.log(className,"**** processContextMenuEvent()-->ce<"+ce+"> ****");
		assert ctxMenuLsnList!=null && ctxMenuLsnList.size()>0;
		Iterator ctxMenuIterator=ctxMenuLsnList.iterator();
		while(ctxMenuIterator.hasNext())
			((ContextMenuListener)ctxMenuIterator.next()).showContextMenu(ce);
	}


   /**
     * Processes history events occurring on this browser
     * by dispatching them to any registered
     * <code>HistoryListener</code> objects.
     * <p>
     * This method is not called unless history events are
     * enabled for this browser. History events are enabled
     * when an <code>HistoryListener</code> object is registered
     * via <code>addHistoryListener</code>.
     *
     * @param       he the History event
     * @return      Boolean appropriate can's depending on event.
     * @see         org.mozilla.jrex.event.history.HistoryListener
     */
    public Boolean processHistoryEvent(HistoryEvent he) {
		if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->he<"+he+"> ****");
		assert hisLsnList!=null &&hisLsnList.size()>0;
		Iterator hisIterator=hisLsnList.iterator();
		int hisEventType=he.getEventType();
		switch(hisEventType){
			case HistoryEventConstants.HIS_NEW_ENTRY_EVENT:
				while(hisIterator.hasNext())
					((HistoryListener)hisIterator.next()).addEntry(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->addEntry ****");
				break;

			case HistoryEventConstants.HIS_GO_BACK_EVENT:
				boolean canGoBack=true;
				while(hisIterator.hasNext())
					canGoBack&=((HistoryListener)hisIterator.next()).goBack(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->goBack canGoBack <"+canGoBack+"> ****");
				return new Boolean(canGoBack);

			case HistoryEventConstants.HIS_GO_FORWARD_EVENT:
				boolean canGoForward=true;
				while(hisIterator.hasNext())
					canGoForward&=((HistoryListener)hisIterator.next()).goForward(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()--> goForward canGoForward <"+canGoForward+"> ****");
				return new Boolean(canGoForward);

			case HistoryEventConstants.HIS_RELOAD_EVENT:
				boolean canReload=true;
				while(hisIterator.hasNext())
					canReload&=((HistoryListener)hisIterator.next()).reload(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->reload canReload <"+canReload+"> ****");
				return new Boolean(canReload);

			case HistoryEventConstants.HIS_GO_TO_INDEX_EVENT:
				boolean canGoToIndex=true;
				while(hisIterator.hasNext())
					canGoToIndex&=((HistoryListener)hisIterator.next()).goToIndex(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->goToIndex canGoToIndex <"+canGoToIndex+"> ****");
				return new Boolean(canGoToIndex);

			case HistoryEventConstants.HIS_PURGE_EVENT:
				boolean canPurge=true;
				while(hisIterator.hasNext())
					canPurge&=((HistoryListener)hisIterator.next()).purge(he);
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()-->purge canPurge <"+canPurge+"> ****");
				return new Boolean(canPurge);

			default:
				if(JRexL.on)JRexL.log(className,"**** processHistoryEvent()--> UNKNOW TYPE ****");
		};
		return null;
	}



   /**
     * Processes URIContent events occurring on this browser
     * by dispatching them to registered
     * <code>URIContentListener</code> object.
     * <p>
     * This method is not called unless URIContent events are
     * enabled for this browser. URIContent events are enabled
     * when an <code>URIContentListener</code> object is registered
     * via <code>addURIContentListener</code>.
     *
     * @param       uce the URIContent event
     * @see         org.mozilla.jrex.event.uricontent.URIContentListener
     */

    public Object processURIContentEvent(URIContentEvent uce) {
		if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()-->uce<"+uce+"> ****");
		assert uriContentListener!=null;
		int uriCntEventType=uce.getEventType();
		switch(uriCntEventType){
			case URIContentEventConstants.URI_ON_START_EVENT:
			{
				boolean abortLoad=uriContentListener.onStartContent(uce);
				if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()-->onStartContent abortLoad <"+abortLoad+"> ****");
				return new Boolean(abortLoad);
			}
			case URIContentEventConstants.URI_DO_CONTENT:
			{
				URIContentEventRV rval=uriContentListener.doContent(uce);
				if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()-->doContent rval <"+rval+"> ****");
				return rval;
			}
			case URIContentEventConstants.URI_IS_PREFERRED:
			{
				URIContentEventRV rval=uriContentListener.isContentPreferred(uce);
				if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()-->isContentPreferred rval <"+rval+"> ****");
				return rval;
			}
			case URIContentEventConstants.URI_CAN_HANDLE_CONTENT: //doesn't isPreferred get the job done?
			{
				URIContentEventRV rval=uriContentListener.canHandleContent(uce);
				if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()-->canHandleContent rval <"+rval+"> ****");
				return rval;
			}
			case URIContentEventConstants.URI_GET_LOAD_COOKIE:
			case URIContentEventConstants.URI_SET_LOAD_COOKIE:
			default:
				if(JRexL.on)JRexL.log(className,"**** processURIContentEvent()--> UNKNOW TYPE ****");
		};
		return null;
    }



	/**
	* Processes TooltipEvent events occurring on this browser
	* @param       tte the TooltipEvent event
	*/
    private static JWindow toolTipWin;
    private static Point location;
    private static JToolTip toolTip;

    public void processTooltipEvent(TooltipEvent tte){
		if(JRexL.on)JRexL.log(className,"**** TooltipEvent()-->tte<"+tte+"> ****");
		int toolTipEventType=tte.getEventType();
		switch(toolTipEventType){
			case TooltipEventConstants.SHOW_EVENT:
				if(toolTipWin==null){
					toolTipWin=new JWindow(SwingUtilities.getWindowAncestor(this));
					toolTip=new JToolTip();
					toolTipWin.getContentPane().add(toolTip);
					location=new Point();
				}
				Point screenLocation = getLocationOnScreen();
				location.setLocation(tte.getClientX()+screenLocation.x,tte.getClientY()+screenLocation.y+ 20);
				toolTip.setTipText(tte.getToolTip());
				Dimension tipSize = toolTip.getPreferredSize();
				Rectangle bounds= getBounds();
				if (location.x <bounds.x)
					location.x =bounds.x;
				else
				if (location.x -bounds.x + tipSize.width >bounds.width)
					location.x = bounds.x + Math.max(0, bounds.width -tipSize.width);

				if (location.y <bounds.y)
					location.y =bounds.y;
				else
				if (location.y -bounds.y + tipSize.height >bounds.height)
					location.y =bounds.y + Math.max(0,bounds.height - tipSize.height);

				toolTipWin.setBounds(location.x,location.y,tipSize.width,tipSize.height);
			    toolTipWin.setVisible(true);
				break;
			case TooltipEventConstants.HIDE_EVENT:
				if(toolTipWin!=null)
					toolTipWin.setVisible(false);
				break;
			default:
				if(JRexL.on)JRexL.log(className,"**** processTooltipEvent()--> UNKNOW TYPE ****");
		};
	}


	/**
	* Invoked to print the current page shown by the browser
	* @param       prompt If <code>true</code> system will prompt with print dialog.
	* @param       showProgress If <code>true</code> print progress dialog is shown.
	*/
	public void print(boolean prompt, boolean showProgress)throws JRexException{
		Print(prompt, showProgress);
	}

	/**
	* Invoked to cancel an ongoing print request.
	* @see	#isPrinting()
	*/
	public void cancelPrint()throws JRexException{
		CancelPrint();
	}

	/**
	* Indicates whether a print is under progress.
	* @see	#cancelPrint()
	* @see	#print(boolean, boolean)
	*/
	public boolean isPrinting()throws JRexException{
		return IsPrinting();
	}

	/**
	* Invoked to Preview the current page shown by the browser
	* @param       shrinkToFit If <code>true</code> Preview will shrink to fit to the browser component view.
	* @param       isLandScape If <code>true</code> Preview will be in LandScape view.
	* @see	#print(boolean, boolean)
	*/

	public void printPreview(boolean shrinkToFit, boolean isLandScape)throws JRexException{
		PrintPreview(shrinkToFit, isLandScape);
	}

	/**
	* Invoked to cancel an print preview.
	* @see	#isPrintPreviewing()
	*/
	public void cancelPrintPreview()throws JRexException{
		CancelPrintPreview();
	}

	/**
	* Indicates whether a print preview is under progress.
	* @see	#cancelPrintPreview()
	* @see	#printPreview(boolean, boolean)
	*/
	public boolean isPrintPreviewing()throws JRexException{
		return IsPrintPreviewing();
	}

	/**
	* Gets the number of print preview pages.
	* @see	#printPreview(boolean, boolean)
	*/

	public int getPrintPreviewNumPages()throws JRexException{
		return GetPrintPreviewNumPages();
	}

	/**
	* Invoked to nagivate in print preview mode.
	* @param	navType the type of navigation to perform.
	* @param	pageNum the page number to navigate to.
	* @see		#printPreview(boolean, boolean)
	* see		PrintPreview Navigation Constants Above.
	*/
	public void printPreviewNavigate(int navType, int pageNum)throws JRexException{
		PrintPreviewNavigate(navType, pageNum);
	}

	/**
	* Invoked to do print PageSetup.
	* @see		#print(boolean, boolean)
	*/
	public void pageSetup()throws JRexException{
		PageSetup();
	}

	/**
	* Indicates whether an undo operation can be done.
	* @see	#undo()
	*/
	public boolean canUndo()throws JRexException{
		return CanUndo();
	}

	/**
	* Indicates whether an redo operation can be done.
	* @see	#redo()
	*/
	public boolean canRedo()throws JRexException{
		return CanRedo();
	}

	/**
	* Indicates whether an cut operation can be done.
	* @see	#cut()
	*/
	public boolean canCut()throws JRexException{
		return CanCut();
	}

	/**
	* Indicates whether an copy operation can be done.
	* @see	#copy()
	*/
	public boolean canCopy()throws JRexException{
		return CanCopy();
	}

	/**
	* Indicates whether an copy linklocation operation can be done.
	* @see	#copyLinkLocation()
	*/
	public boolean canCopyLinkLocation()throws JRexException{
		return CanCopyLinkLocation();
	}

	/**
	* Indicates whether an copy Imagelocation operation can be done.
	* @see	#copyImageLocation()
	*/
	public boolean canCopyImageLocation()throws JRexException{
		return CanCopyImageLocation();
	}

	/**
	* Indicates whether an copy ImageContents operation can be done.
	* @see	#copyImageContents()
	*/
	public boolean canCopyImageContents()throws JRexException{
		return CanCopyImageContents();
	}


	/**
	* Indicates whether an Paste operation can be done.
	* @see	#paste()
	*/
	public boolean canPaste()throws JRexException{
		return CanPaste();
	}

	/**
	* Indicates whether an Delete operation can be done.
	* @see	#delete()
	*/
	public boolean canDelete()throws JRexException{
		return CanDelete();
	}

	/**
	* Invoked to undo an edit operation.
	*/
	public void undo()throws JRexException{
		Undo();
	}

	/**
	* Invoked to redo an edit operation.
	*/
	public void redo()throws JRexException{
		Redo();
	}

	/**
	* Invoked to cut selected text.
	*/
	public void cut()throws JRexException{
		Cut();
	}

	/**
	* Invoked to copy selected text.
	*/
	public void copy()throws JRexException{
		Copy();
	}

	/**
	* Invoked to copy selected LinkLocation.
	*/
	public void copyLinkLocation()throws JRexException{
		CopyLinkLocation();
	}

	/**
	* Invoked to copy selected ImageLocation.
	*/
	public void copyImageLocation()throws JRexException{
		CopyImageLocation();
	}

	/**
	* Invoked to copy ImageContents.
	*/
	public void copyImageContents()throws JRexException{
		CopyImageContents();
	}

	/**
	* Invoked to paste text from clipboard.
	*/
	public void paste()throws JRexException{
		Paste();
	}

	/**
	* Invoked to select all text in an control.
	*/
	public void selectAll()throws JRexException{
		SelectAll();
	}

	/**
	* Invoked to clear all selection in an control.
	*/
	public void selectNone()throws JRexException{
		SelectNone();
	}

	/**
	* Invoked to delete selected text.
	*/
	public void delete()throws JRexException{
		Delete();
	}

	/**
	* Invoked to find a given text.
	*/
	public void find()throws JRexException{
		Find();
	}

	/**
	* Invoked to repeat previous find.
	*/
	public void findAgain()throws JRexException{
		FindAgain();
	}

	private static native void initIDs()throws JRexException;
	private native void CreatePeer(int waitKey)throws JRexException;
	private native static void AbortPeerCreation(int waitKey)throws JRexException;
	private native void GotVisual()throws JRexException;
	private native void RealizePeer()throws JRexException;
	private native void DestroyPeer()throws JRexException;
	private native void SetBoundsPeer()throws JRexException;
	private native Rectangle GetBoundsPeer()throws JRexException;
	private native void SetVisiblePeer(boolean visible)throws JRexException;
	private native void SetEnabledPeer(boolean visible)throws JRexException;
	private native boolean IsVisiblePeer()throws JRexException;
	private native boolean IsEnabledPeer()throws JRexException;
	private native String GetTitlePeer()throws JRexException;
	private native void SetTitlePeer(String title)throws JRexException;
	private native void RepaintPeer(boolean force)throws JRexException;
	private native void SetFocusPeer(boolean focusGained)throws JRexException;

	private native void Print(boolean prompt, boolean showProgress)throws JRexException;
	private native void CancelPrint()throws JRexException;
	private native boolean IsPrinting()throws JRexException;
	private native void PrintPreview(boolean shrinkToFit, boolean isLandScape)throws JRexException;
	private native void CancelPrintPreview()throws JRexException;
	private native int GetPrintPreviewNumPages()throws JRexException;
	private native boolean IsPrintPreviewing()throws JRexException;
	private native void PrintPreviewNavigate(int navType, int pageNum)throws JRexException;

	private native void PageSetup()throws JRexException;


	private native boolean CanUndo();
	private native boolean CanRedo();
	private native boolean CanCut();
	private native boolean CanCopy();
	private native boolean CanCopyLinkLocation();
	private native boolean CanCopyImageLocation();
	private native boolean CanCopyImageContents();
	private native boolean CanPaste();
	private native boolean CanDelete();

	private native void Undo();
	private native void Redo();
	private native void Cut();
	private native void Copy();
	private native void CopyLinkLocation();
	private native void CopyImageLocation();
	private native void CopyImageContents();
	private native void Paste();
	private native void SelectAll();
	private native void SelectNone();
	private native void Delete();

	private native void Find()throws JRexException;
	private native void FindAgain()throws JRexException;

	private native void AddProgressListener()throws JRexException;
	private native void AddContextMenuListener()throws JRexException;
	private native void AddHistoryListener()throws JRexException;
	private native void AddKeyListener()throws JRexException;
	private native void AddMouseListener()throws JRexException;
	private native void AddMouseMotionListener()throws JRexException;
	private native void AddURIContentListener()throws JRexException;

	private native void RemoveProgressListener()throws JRexException;
	private native void RemoveContextMenuListener()throws JRexException;
	private native void RemoveHistoryListener()throws JRexException;
	private native void RemoveKeyListener()throws JRexException;
	private native void RemoveMouseListener()throws JRexException;
	private native void RemoveMouseMotionListener()throws JRexException;
	private native void RemoveURIContentListener()throws JRexException;
}