#include "HRTimer.h"
#include "nsMemory.h"
#include "nsVoidArray.h"
#include "nsCOMPtr.h"
#include "nsXPCOMCID.h"
#include "nsIVariant.h"
#include "nsComponentManagerUtils.h"
#include "nsServiceManagerUtils.h"
#include "nsIServiceManager.h"
#include <time.h>
#include "common.h"

#define HAVE_POSIX_TIMERS (defined(_POSIX_TIMERS) && _POSIX_TIMERS != -1)
#define HAVE_OTHER_POSIX (HAVE_UNISTD_H && !defined(WIN32))

#if HAVE_OTHER_POSIX
#include <unistd.h>
/* For getrusage and gettimeofday */
#include <sys/time.h>
#include <sys/resource.h>

/* For POSIX high-resolution timers, if present */
#include <time.h>

inline static PRUint64 extract_times(const struct timeval* tv)
{
  return tv->tv_sec * 1000000000LL + tv->tv_usec * 1000LL;
}

inline static PRUint64 extract_times(const struct timespec* ts)
{
  return (PRUint64)ts->tv_sec * 1000000000LL + (PRUint64)ts->tv_nsec;
}
#endif // HAVE_OTHER_POSIX

#ifdef WIN32
#include <windows.h>
#include <Psapi.h>
#endif // WIN32

class HRTimerBase : public IHRTimer {
  public:
  HRTimerBase(const char* name, bool is_counter);
  ~HRTimerBase();

  NS_SCRIPTABLE NS_IMETHOD GetName(char * *aName);
  NS_SCRIPTABLE NS_IMETHOD GetIsCounter(PRBool* out);

  private:
  char* mName;
  bool mIsCounter;
};

#define IMPL_TIMER_SUPPORTS(_clsname)                           \
  NS_IMPL_ISUPPORTS2(_clsname, IHRTimer, nsIClassInfo)          \
  NS_IMPL_CI_INTERFACE_GETTER1(_clsname, IHRTimer)              \
  IMPL_CI(_clsname, # _clsname, nsIClassInfo::DOM_OBJECT)

HRTimerBase::HRTimerBase(const char* name, bool is_counter) :
  mName(nsstrdup(name)), mIsCounter(is_counter)
{}

HRTimerBase::~HRTimerBase() {
  nsMemory::Free(mName);
}

NS_IMETHODIMP HRTimerBase::GetName(char** out) {
  *out = nsstrdup(mName);
  if(!*out) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  return NS_OK;
}

NS_IMETHODIMP HRTimerBase::GetIsCounter(PRBool* out) {
  *out = mIsCounter;
  return NS_OK;
}

#if HAVE_POSIX_TIMERS
/* Posix-timers */
class HRPosixTimer : public HRTimerBase, public nsIClassInfo {
  public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  HRPosixTimer(const char* name, clockid_t clock);

  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRPosixTimer();
  clockid_t mClockID;
};

IMPL_TIMER_SUPPORTS(HRPosixTimer)

HRPosixTimer::HRPosixTimer(const char* name, clockid_t clock)
: HRTimerBase(name, false), mClockID(clock) {}

HRPosixTimer::~HRPosixTimer() {}

/* void getTime (out unsigned long seconds, out unsigned long nanoseconds); */
NS_IMETHODIMP HRPosixTimer::GetTime(PRUint64 *nanoseconds)
{
  struct timespec ts;
  if(clock_gettime(mClockID, &ts)) {
    return NS_ERROR_FAILURE;
  }

  *nanoseconds = extract_times(&ts);
  return NS_OK;
}

/* readonly attribute unsigned long resolution; */
NS_IMETHODIMP HRPosixTimer::GetResolution(PRUint32 *aResolution)
{
  struct timespec ts;
  if(clock_getres(mClockID, &ts)) {
    return NS_ERROR_FAILURE;
  }

  *aResolution = ts.tv_nsec;
  return NS_OK;
}

static struct { const char* name; clockid_t id; } avail_posix_timers[] =
{
#ifdef CLOCK_REALTIME
  { "CLOCK_REALTIME", CLOCK_REALTIME },
#endif

#ifdef CLOCK_MONOTONIC
  { "CLOCK_MONOTONIC", CLOCK_MONOTONIC },
#endif

#ifdef CLOCK_PROCESS_CPUTIME_ID
  { "CLOCK_PROCESS_CPUTIME_ID", CLOCK_PROCESS_CPUTIME_ID },
#endif

#ifdef CLOCK_THREAD_CPUTIME_ID
  { "CLOCK_THREAD_CPUTIME_ID", CLOCK_THREAD_CPUTIME_ID },
#endif

#ifdef CLOCK_REALTIME_HR
  { "CLOCK_REALTIME_HR", CLOCK_REALTIME_HR },
#endif

#ifdef CLOCK_MONOTONIC_HR
  { "CLOCK_MONOTONIC_HR", CLOCK_MONOTONIC_HR },
#endif

  { NULL, 0 }
};

#define NUM_POSIX_TIMERS                                                \
  ((sizeof(avail_posix_timers)/sizeof(*avail_posix_timers)) - 1)

#endif // HAVE_POSIX_TIMERS

#if HAVE_OTHER_POSIX

class HRGetTimeOfDayTimer : public HRTimerBase, nsIClassInfo {
  public:

  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  HRGetTimeOfDayTimer(const char* name);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRGetTimeOfDayTimer();
};

IMPL_TIMER_SUPPORTS(HRGetTimeOfDayTimer)

NS_IMETHODIMP HRGetTimeOfDayTimer::GetTime(PRUint64* nanoseconds)
{
  struct timeval tv;
  if(gettimeofday(&tv, NULL)) {
    return NS_ERROR_FAILURE;
  }

  *nanoseconds = extract_times(&tv);
  return NS_OK;
}

NS_IMETHODIMP HRGetTimeOfDayTimer::GetResolution(PRUint32* nanoseconds)
{
  *nanoseconds = (PRUint32)(1000000000LL / sysconf(_SC_CLK_TCK));
  return NS_OK;
}

HRGetTimeOfDayTimer::HRGetTimeOfDayTimer(const char* name)
  : HRTimerBase(name, false)
{}

HRGetTimeOfDayTimer::~HRGetTimeOfDayTimer() {}


class HRTimesTimer : public HRTimerBase, nsIClassInfo {
  public:

  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO
  // POSIX guarantees these two exist
  enum rusage_timer_t { USER, SYSTEM };

  // Copies name
  HRTimesTimer(const char* name, rusage_timer_t which);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);


  private:
  ~HRTimesTimer();
  rusage_timer_t mWhichTime;
};

IMPL_TIMER_SUPPORTS(HRTimesTimer)

NS_IMETHODIMP HRTimesTimer::GetTime(PRUint64* nanoseconds)
{
  struct rusage r;
  if(getrusage(RUSAGE_SELF, &r)) {
    return NS_ERROR_FAILURE;
  }

  if(mWhichTime == USER) {
    *nanoseconds = extract_times(&r.ru_utime);
  } else {
    PR_ASSERT(mWhichTime == SYSTEM);
    *nanoseconds = extract_times(&r.ru_stime);
  }

  return NS_OK;
}

NS_IMETHODIMP HRTimesTimer::GetResolution(PRUint32* nanoseconds)
{
  *nanoseconds = (PRUint32)(1000000000LL / sysconf(_SC_CLK_TCK));
  return NS_OK;
}

HRTimesTimer::HRTimesTimer(const char* name, rusage_timer_t which)
  : HRTimerBase(name, false), mWhichTime(which) {}
HRTimesTimer::~HRTimesTimer() {}

class HRRUsageCounterTimer : public HRTimerBase, nsIClassInfo {
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  public:
  // Copies name
  HRRUsageCounterTimer(const char* name, size_t mFieldOffset);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRRUsageCounterTimer();
  size_t mFieldOffset;
};

IMPL_TIMER_SUPPORTS(HRRUsageCounterTimer)

NS_IMETHODIMP HRRUsageCounterTimer::GetTime(PRUint64* nanoseconds)
{
  struct rusage r;
  if(getrusage(RUSAGE_SELF, &r)) {
    return NS_ERROR_FAILURE;
  }

  long* value = (long*)((char*)&r + mFieldOffset);
  *nanoseconds = *value;
  return NS_OK;
}

NS_IMETHODIMP HRRUsageCounterTimer::GetResolution(
  PRUint32* nanoseconds)
{
  *nanoseconds = 1000*1000*1000;
  return NS_OK;
}

HRRUsageCounterTimer::HRRUsageCounterTimer(const char* name, size_t fieldOffset)
  : HRTimerBase(name, true), mFieldOffset(fieldOffset)
{
  PR_ASSERT(fieldOffset <= sizeof(struct rusage) - sizeof(long));
}

HRRUsageCounterTimer::~HRRUsageCounterTimer() {}

static struct { const char* name; size_t offset; } avail_rusage_counters[] =
{
  { "getrusage()-Maximum RSS", offsetof(struct rusage, ru_maxrss) },
  { "getrusage()-Page Reclaims", offsetof(struct rusage, ru_minflt) },
  { "getrusage()-Page Faults", offsetof(struct rusage, ru_majflt) },
  { "getrusage()-Swaps", offsetof(struct rusage, ru_nswap) },
  { "getrusage()-Blocks Read", offsetof(struct rusage, ru_inblock) },
  { "getrusage()-Blocks Written",
    offsetof(struct rusage, ru_oublock) },
  { "getrusage()-Signals Received",
    offsetof(struct rusage, ru_nsignals) },
  { "getrusage()-Voluntary Context Switches",
    offsetof(struct rusage, ru_nvcsw) },
  { "getrusage()-Involuntary Context Switches",
    offsetof(struct rusage, ru_nivcsw) },
  {NULL, 0}
};

#define NUM_RUSAGE_COUNTERS                                             \
  ((sizeof(avail_rusage_counters) / sizeof(*avail_rusage_counters)) - 1)

#endif // HAVE_OTHER_POSIX

#ifdef WIN32

static inline PRUint64 extract_times(const FILETIME* ft)
{
  return 100 * ( (PRUint64)ft->dwLowDateTime
                 | ((PRUint64)ft->dwHighDateTime << 32) );
}

class HRThreadTimesTimer : public HRTimerBase, nsIClassInfo {
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  public:

  enum thread_time_t { USER, KERNEL };

  // Copies name
  HRThreadTimesTimer(const char* name, thread_time_t time_type);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRThreadTimesTimer();

  thread_time_t mTimeType;
};

IMPL_TIMER_SUPPORTS(HRThreadTimesTimer)

HRThreadTimesTimer::HRThreadTimesTimer(const char* name,
                                       thread_time_t time_type)
: HRTimerBase(name, false),
mTimeType(time_type)
{}

HRThreadTimesTimer::~HRThreadTimesTimer() {}

NS_IMETHODIMP HRThreadTimesTimer::GetResolution(PRUint32* nanoseconds) {
  // NT kernels fire timer every 10ms. (GetThreadTimes is very
  // coarse.)
  *nanoseconds = 10000000;
  return NS_OK;
}

NS_IMETHODIMP HRThreadTimesTimer::GetTime(PRUint64* nanoseconds) {
  FILETIME creation_time, exit_time, user_time, kernel_time;

  if(!GetThreadTimes(GetCurrentThread(),
                     &creation_time, &exit_time, &user_time, &kernel_time))
  {
    return NS_ERROR_FAILURE;
  }

  if(mTimeType == HRThreadTimesTimer::USER) {
    *nanoseconds = extract_times(&user_time);
  } else {
    PR_ASSERT(mTimeType == HRThreadTimesTimer::KERNEL);
    *nanoseconds = extract_times(&kernel_time);
  }

  return NS_OK;
}

class HRQueryPerformanceCounterTimer : public HRTimerBase, nsIClassInfo {
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  public:

  // Copies name
  HRQueryPerformanceCounterTimer(const char* name);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  static bool IsSupported();

  private:
  ~HRQueryPerformanceCounterTimer();
  PRUint64 mQPCHz; // 0 if unsupported
};

IMPL_TIMER_SUPPORTS(HRQueryPerformanceCounterTimer)

bool HRQueryPerformanceCounterTimer::IsSupported() {
  LARGE_INTEGER dummy;
  return QueryPerformanceFrequency(&dummy);
}

HRQueryPerformanceCounterTimer::HRQueryPerformanceCounterTimer(const char* name)
  : HRTimerBase(name, false)
{
  LARGE_INTEGER frequency;
  QueryPerformanceFrequency(&frequency);
  mQPCHz = frequency.QuadPart;
}

HRQueryPerformanceCounterTimer::~HRQueryPerformanceCounterTimer() {}

NS_IMETHODIMP HRQueryPerformanceCounterTimer::GetTime(PRUint64* nanoseconds)
{
  LARGE_INTEGER t;
  if(!QueryPerformanceCounter(&t)) {
    return NS_ERROR_FAILURE;
  }

  *nanoseconds = (1000000000LL * t.QuadPart) / mQPCHz;

  return NS_OK;
}

NS_IMETHODIMP HRQueryPerformanceCounterTimer::GetResolution(PRUint32* nanoseconds)
{
  PRUint64 res = (PRUint64)1000000000 / mQPCHz;
  if(res > PR_UINT32_MAX) {
    *nanoseconds = PR_UINT32_MAX;
  } else {
    *nanoseconds = (PRUint32)res;
  }

  return NS_OK;
}


class HRPageFaultTimer : public HRTimerBase, nsIClassInfo {
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  public:

  // Copies name
  HRPageFaultTimer(const char* name);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRPageFaultTimer();
};

IMPL_TIMER_SUPPORTS(HRPageFaultTimer)

HRPageFaultTimer::HRPageFaultTimer(const char* name)
: HRTimerBase(name, true)
{}

HRPageFaultTimer::~HRPageFaultTimer() {}

NS_IMETHODIMP HRPageFaultTimer::GetTime(PRUint64* nanoseconds)
{
  PROCESS_MEMORY_COUNTERS counters;
  if(!GetProcessMemoryInfo(GetCurrentProcess(),
                           &counters,
                           sizeof(counters)))
  {
    return NS_ERROR_FAILURE;
  }

  *nanoseconds = counters.PageFaultCount;
  return NS_OK;
}

NS_IMETHODIMP HRPageFaultTimer::GetResolution(PRUint32* nanoseconds)
{
  *nanoseconds = 1;
  return NS_OK;
}

class HRPeakPagefileUseTimer : public HRTimerBase, nsIClassInfo {
  NS_DECL_ISUPPORTS
  NS_DECL_NSICLASSINFO

  public:

  // Copies name
  HRPeakPagefileUseTimer(const char* name);
  NS_SCRIPTABLE NS_IMETHOD GetTime(PRUint64* nanoseconds);
  NS_SCRIPTABLE NS_IMETHOD GetResolution(PRUint32* nanoseconds);

  private:
  ~HRPeakPagefileUseTimer();
};

IMPL_TIMER_SUPPORTS(HRPeakPagefileUseTimer)

HRPeakPagefileUseTimer::HRPeakPagefileUseTimer(const char* name)
: HRTimerBase(name, true)
{}

HRPeakPagefileUseTimer::~HRPeakPagefileUseTimer() {}

NS_IMETHODIMP HRPeakPagefileUseTimer::GetTime(PRUint64* nanoseconds)
{
  PROCESS_MEMORY_COUNTERS counters;
  if(!GetProcessMemoryInfo(GetCurrentProcess(),
                           &counters,
                           sizeof(counters)))
  {
    return NS_ERROR_FAILURE;
  }

  *nanoseconds = counters.PeakPagefileUsage;
  return NS_OK;
}

NS_IMETHODIMP HRPeakPagefileUseTimer::GetResolution(PRUint32* nanoseconds)
{
  *nanoseconds = 1;
  return NS_OK;
}


#endif // WIN32

// ---------- timer factory --------------

NS_IMPL_ISUPPORTS1_CI(HRTimerFactory, IHRTimerFactory)

HRTimerFactory::HRTimerFactory() {
#if HAVE_POSIX_TIMERS
  for(unsigned i = 0; i < NUM_POSIX_TIMERS; ++i) {
    mTimers.AppendObject( // addrefs
      new HRPosixTimer(avail_posix_timers[i].name,
                       avail_posix_timers[i].id));
  }
#endif // HAVE_POSIX_TIMERS

#if HAVE_OTHER_POSIX // POSIX
  mTimers.AppendObject( // addrefs
    new HRGetTimeOfDayTimer("gettimeofday()"));

  mTimers.AppendObject(
    new HRTimesTimer("getrusage()-User Time", HRTimesTimer::USER));

  mTimers.AppendObject(
    new HRTimesTimer("getrusage()-System Time", HRTimesTimer::SYSTEM));

  for(unsigned i = 0; i < NUM_RUSAGE_COUNTERS; ++i) {
    mTimers.AppendObject(
      new HRRUsageCounterTimer(avail_rusage_counters[i].name,
                               avail_rusage_counters[i].offset));
  }
#endif // HAVE_OTHER_POSIX

#ifdef WIN32
  if(HRQueryPerformanceCounterTimer::IsSupported()) {
    mTimers.AppendObject(
      new HRQueryPerformanceCounterTimer("QueryPerformanceCounter()"));
  }

  mTimers.AppendObject(
    new HRThreadTimesTimer("GetThreadTimes()-User Time",
                           HRThreadTimesTimer::USER));

  mTimers.AppendObject(
    new HRThreadTimesTimer("GetThreadTimes()-Kernel Time",
                           HRThreadTimesTimer::KERNEL));

  mTimers.AppendObject(
    new HRPageFaultTimer("ProcessMemoryInfo-PageFaultCount"));

  mTimers.AppendObject(
    new HRPeakPagefileUseTimer("ProcessMemoryInfo-PeakPagefileUsage"));

#endif // WIN32

}

HRTimerFactory::~HRTimerFactory() {}

/* void getAvailableTimers
 * (out unsigned long num,
 * [array, size_is (num), retval] out string timers); */
NS_IMETHODIMP HRTimerFactory::GetAvailableTimers(
  PRUint32 *num, char ***timers)
{
  *num = mTimers.Count();
  *timers = (char**)nsMemory::Alloc(*num * sizeof(**timers));
  if(!*timers) {
    return NS_ERROR_OUT_OF_MEMORY;
  }

  for(unsigned i = 0; i < *num; ++i) {
    nsresult rv = mTimers[i]->GetName(&(*timers)[i]);
    NS_ENSURE_SUCCESS(rv, rv);
  }

  return NS_OK;
}

/* IHRTimer makeTimer (in string timer_name); */
NS_IMETHODIMP HRTimerFactory::MakeTimer(
  const char* timer_name, IHRTimer **_retval)
{
  unsigned num = mTimers.Count();
  for(unsigned i = 0; i < num; ++i) {
    nsCString str;
    nsresult rv = mTimers[i]->GetName(getter_Copies(str));
    NS_ENSURE_SUCCESS(rv, rv);
    if(str.Equals(timer_name)) {
      *_retval = mTimers[i];
      NS_ADDREF(*_retval);
      break;
    }
  }

  return NS_OK;
}


