function report_sql_error(db, msg) {
  throw new Error(msg + ': ' +
                  db.lastError + ' - ' +
                  db.lastErrorString);
}

function safe_create_wrapped_statement(db, sql) {
  try {
    var stmt = db.createStatement(sql);
  } catch(e) {
    report_sql_error(db, 'error creating statment "' +
                     sql + '"');
  }

  var wrapped = gStatementWrapper.createInstance(Ci.mozIStorageStatementWrapper);
  wrapped.initialize(stmt);
  return wrapped;
}

function SafeStmtWrapper(db, sql) {
  this._db = db;
  this._sql = sql;
  this._wrapped = safe_create_wrapped_statement(db, sql);
  this._stmt = this._wrapped.statement;
}

SafeStmtWrapper.prototype = {
  constructor: SafeStmtWrapper,
  
  _bind: function(params) {
    for(var name in params) {
      this._wrapped.params[name] = params[name];
    }
  },

  _copyRow: function(row) {
    var num = this._stmt.columnCount;
    var res = {};
    
    for(var i = 0; i < num; ++i) {
      var name = this._stmt.getColumnName(i);
      res[name] = row[name];
    }

    return res;
  },

  finalize: function() {
    this._stmt.finalize();
  },
  
  execute: function(params) {
    if(params) {
      this._bind(params);
    }

    try {
      this._wrapped.execute();
    } catch(e) {
      this._wrapped.reset();
      report_sql_error(this._db, 'query: "' + this._sql + '"');
    }
  },

  queryOneCol: function(col, params) {
    if(params) {
      this._bind(params);
    }
    
    try {
      try {
        var has_row = this._wrapped.step();
      } catch(e) {
        report_sql_error(this._db, 'query: "' + this._sql + '"');
      }

      if(has_row) {
        return this._wrapped.row[col];
      }

      return null;
    } finally {
      this._wrapped.reset();
    }    
  },

  queryOneRow: function(params) {
    if(params) {
      this._bind(params);
    }

    try {
      try {
        var has_row = this._wrapped.step();
      } catch(e) {
        report_sql_error(this._db, 'query: "' + this._sql + '"');
      }

      if(has_row) {
        return this._copyRow(this._wrapped.row);
      }

      return null;
      
    } finally {
      this._wrapped.reset();
    }
  },

  queryAllRows: function(params) {
    var result = [];

    if(params) {
      this._bind(params);
    }

    /*
    dump('queryAllRows:' + this._sql +
         (params ? ' params:' + params.toSource() : '')
         + '\n');
    */
    
    try {
      for(;;) {
        try {
          var has_row = this._wrapped.step();
        } catch(e) {
          report_sql_error(this._db, 'query: "' + this._sql + '"');
        }
        
        if(has_row) {
          result.push(this._copyRow(this._wrapped.row));
        } else {
          break;
        }
      }
    } finally {
      this._wrapped.reset();
    }

    return result;
  }
}

/**
  * Call database for a profile run.
  * 
  * @param input: file to use for the database
  * @param conv: Call log to convert
  */
function HRPCallTree(file) {
  var db = this._db = gStorageService.openDatabase(file);

  // Must do this before anything else
  this._safeSQL('PRAGMA page_size=4096');
  this._safeSQL('PRAGMA read_uncommitted=1');
  
  this._cachedQueries = {};
  this._createTables();
  this._inTransaction = false;
  this._infoByID = false;

  if(!this._readMetaData()) {
    this._setMetaData({version: 1});
  }
}

HRPCallTree.prototype = {
  constructor: HRPCallTree,
  
  _metaDataNames: ['searchedChromeWindow',
                   'searchedWindow',
                   'searchedOther',
                   'searchedOnlyPrototypes',
                   'preferredGivenNames',
                   'usedFileLineCache',
                   'maxDepth',
                   'mainThreadOnly',
                   'timeSource',
                   'timeSourceIsCounter',
                   'version'],
  
  _ensureNoTransaction: function() {
    if(this._inTransaction) {
      this._safeSQL('COMMIT');
      this._inTransaction = false;
    }
  },
  
  _ensureTransaction: function() {
    if(!this._inTransaction) {
      this._safeSQL('BEGIN IMMEDIATE TRANSACTION');
      this._inTransaction = true;
    }
  },

  _setMetaData: function(metadata) {
    this._getOrCacheQuery('DELETE FROM metadata').execute();
    var keys = [];
    
    for(var key in metadata) {
      keys.push(key);
    }
    
    var sql = 'INSERT INTO metadata (' + keys.join(',') + ')' +
      ' VALUES (' + keys.map(function(k) ':' + k).join(',') + ')';
    
    this._getOrCacheQuery(sql).execute(metadata);
    this._readMetaData();
  },

  _readMetaData: function() {
    this._metaData = this._getOrCacheQuery('SELECT * FROM metadata')
      .queryOneRow();
    return this._metaData;
  },

  _getInfoByID: function() {
    if(!this._infoByID) {
      // Use the funcid-sorted flat profile so we append to the new
      // info array contiguously, hopefully allowing spidermonkey to
      // use the dense representation
      var results = this.getFlatProfile('funcid');
      var info = this._infoByID = [];
      for(var i = 0; i < results.length; ++i) {
        var row = results[i];
        info[row.funcid] = row;
      }
    }
    
    return this._infoByID;
  },
  
  /**
    * Close the underlying database connection. The HRPCallTree cannot
    * be used after it is closed. Multiple calls to close are fine.
    *
    * @return: no return value
    */
  close: function() {
    if(this._db) {
      this._ensureNoTransaction();
      
      for(var sql in this._cachedQueries) {
        var stmt = this._cachedQueries[sql];
        stmt.finalize();
      }

      this._cachedQueries = null;
      this._db.close();
      this._db = null;
    }
  },
  
  _safeSQL: function(sql) {
    try {
      this._db.executeSimpleSQL(sql);
    } catch(e) {
      report_sql_error(this._db, 'query: "' + sql + '"');
    }
  },

  _safeDummySQL: function(sql) {
    try {
      this._dummyDB.executeSimpleSQL(sql);
    } catch(e) {
      report_sql_error(this._dummyDB, 'dummy query: "' + sql + '"');
    }

  },

  /**
    * Caller is responsible for calling finalize() on the statement
    */
  _getUncachedQuery: function(sql) {
    return new SafeStmtWrapper(this._db, sql);
  },

  _getOrCacheQuery: function(sql) {
    if(sql in this._cachedQueries) {
      return this._cachedQueries[sql];
    }

    return this._addManualCachedQuery(sql, sql);
  },

  /**
    * Add a compiled statement to be finalized at close()-time. Return
    * the compiled statement.
    */
  _addManualCachedQuery: function(tag, sql) {
    var stmt = new SafeStmtWrapper(this._db, sql);
    var old_stmt = this._cachedQueries[tag];

    if(old_stmt !== undefined) {
      old_stmt.finalize();
    }
    
    this._cachedQueries[tag] = stmt;
    return stmt;
  },

  /**
    * Find the maximum ID in use in the calls table. Return 0 if no
    * entries exist.
    */
  _getMaxCallID: function() {
    return this._getOrCacheQuery('SELECT MAX(id) AS id FROM calls')
      .queryOneCol('id') || 0;
  },

  _getMaxCallLevel: function() {
    return this._getOrCacheQuery('SELECT MAX(level) AS level FROM calls')
      .queryOneCol('level') || 0;
  },
  
  // Find function, return null if not found
  _findFunction: function(name, filename, lineno) {
    var stmt = this._getOrCacheQuery(
      ('SELECT id'                    +
       '  FROM functions'             +
       ' WHERE name = :name'          +
       '   AND filename = :filename ' +
       '   AND lineno = :lineno'));

    return stmt.queryOneCol('id',
                            {name: name,
                             filename: filename,
                             lineno: lineno});
  },

  _createFunction: function(name, filename, lineno) {
    assert(function() typeof name === 'string');
    assert(function() typeof filename === 'string');
    assert(function() typeof lineno === 'number');
    
    var stmt = this._getOrCacheQuery(
      ('INSERT INTO functions (name, filename, lineno)' +
       '     VALUES (:name, :filename, :lineno)'));
    
    stmt.execute({
      name: name,
      filename: filename,
      lineno: lineno});
    
    return this._db.lastInsertRowID;
  },
  
  _findOrCreateFunction: function(name, filename, lineno) {
    var id = this._findFunction(name, filename, lineno);
    if(!id) {
      id = this._createFunction(name, filename, lineno);
    }

    assert(function() id);
    return id;    
  },
  
  /**
    * Create the tables
    */
  _createTables: function() {
    var db = this._db;

    // Single-row table.
    this._safeSQL(
      ('CREATE TABLE IF NOT EXISTS metadata (                 ' +
       // File version
       ' version INTEGER,                                     ' +
       // Profiling settings
       ' searchedChromeWindow BOOLEAN,                        ' +
       ' searchedWindow BOOLEAN,                              ' +
       ' searchedOther BOOLEAN,                               ' +
       ' searchedOnlyPrototypes BOOLEAN,                      ' +
       ' preferredGivenNames BOOLEAN,                         ' +
       ' usedFileLineCache BOOLEAN,                           ' +
       ' maxDepth INTEGER,                                    ' +
       ' mainThreadOnly BOOLEAN,                              ' +
       ' timeSource TEXT,                                     ' +
       ' timeSourceIsCounter BOOLEAN)                         ' )
    );

    this._safeSQL(
      ('CREATE TABLE IF NOT EXISTS functions ('                 +
       ' id         INTEGER PRIMARY KEY AUTOINCREMENT,'         +
       ' name       TEXT NOT NULL,'                             +
       ' filename   TEXT NOT NULL,'                             +
       ' lineno     INTEGER NOT NULL)'));

    this._safeSQL(
      ('CREATE TABLE IF NOT EXISTS calls (                     ' +
       ' id                 INTEGER PRIMARY KEY AUTOINCREMENT, ' +
       ' thread             INTEGER NOT NULL,                  ' +
       ' caller             INTEGER NULL,                      ' +
       ' func               INTEGER NOT NULL,                  ' +
       ' start              REAL NOT NULL,                     ' +
       ' end_1              REAL NOT NULL,                     ' +
       ' end_2              REAL NOT NULL,                     ' +
       ' level              INTEGER NOT NULL,                  ' +
       ' seconds_self       REAL NOT NULL,                     ' +
       ' seconds_total      REAL NOT NULL,                     ' +
       // seconds_dbg_self would just be end_2 - end_1
       ' seconds_dbg_total  REAL NOT NULL)                     '
      ));

    this._safeSQL(
      ('CREATE TABLE IF NOT EXISTS cached_profile_results (    ' +
       ' query              TEXT NOT NULL,                     ' +
       ' name               TEXT NOT NULL,                     ' +
       ' filename           TEXT NOT NULL,                     ' +
       ' lineno             INTEGER NOT NULL,                  ' +
       ' funcid             INTEGER NOT NULL,                  ' +
       ' seconds_total      REAL NOT NULL,                     ' +
       ' seconds_self       REAL NOT NULL,                     ' +
       ' seconds_dbg_self   REAL NOT NULL,                     ' +
       ' seconds_dbg_total  REAL NOT NULL,                     ' +
       ' calls              INTEGER NOT NULL)                  ' )
    );

    this._safeSQL(
      ('CREATE INDEX IF NOT EXISTS                             ' +
       ' cached_profile_results_query                          ' +
       ' ON cached_profile_results(query)                      ' )
    );

    this._safeSQL(
      ('CREATE UNIQUE INDEX IF NOT EXISTS                      ' +
       ' functions_name_idx ON functions (name, filename, lineno)')
    );

    this._safeSQL('CREATE INDEX IF NOT EXISTS calls_func_idx ON calls(func) ');

    this._safeSQL(
      ('CREATE INDEX IF NOT EXISTS                             ' +
       ' calls_caller_idx ON calls (caller)                    ' )
    );
  },
  
  /**
    * Add the information in the call log to this database
    *
    * @param input_file: nsIInputSTream
    * @param chunksize: number of records to process before yielding
    * @return: iterator
    */
  parseCallLog: function(input, chunksize) {
    assert(function() input instanceof Ci.nsIInputStream);
    assert(function() typeof chunksize === 'number');
    
    /* SQLite is most efficient when inserting keys in-order. The
     * problem is that we can't insert a complete call record until we
     * encounter the *return* record, and because a complete call
     * record must refer to its caller, we're forced to insert rows
     * with out-of-order IDs.
     *
     * Consider the following sequence of records:
     *
     * A: CALL-foo
     * B: CALL-bar
     * C: CALL-qux
     * D: RET-qux
     * E: RET-bar
     * F: RET-foo
     *
     * We'd like to create three call records:
     *
     * { id: 1, func: foo, caller: null }
     * { id: 2, func: bar, caller: 1    }
     * { id: 3, func: qux, caller: 2    }
     * 
     * The problem is that we can't insert a call record until we see
     * the RET, but that record needs to refer to its parent. We can
     * assign an ID when we see the CALL and use that ID as the caller
     * when we insert the RET, but that results in sub-optimal
     * out-of-order insertion. In the example above, we assign IDs 1,
     * 2, and 3 as we encounter the CALL records. Then we insert
     * functions 3, 2, and 1, with callers 2, 1, and null,
     * respectively. While correct, that ordering is suboptimal with
     * respect to index performance.
     *
     * We hack around this problem by using a temporary table with no
     * indexes whatsoever. We iterate through our input file (once!)
     * inserting entries as we go to this unindexed table. Then we
     * bulk-add the contents of the temporary table to the real calls
     * table.
     *
     */

    var entry = gHrp.parseProfilerData(input);
    var main_thread_only = entry.mainThreadOnly;

    // column names and order must be synchronized with those of table
    // ‘calls’ in _createTables.
    this._safeSQL(
      ('CREATE TEMPORARY TABLE temp.unindexed_calls (' +
       ' id                INTEGER NOT NULL,         ' +
       ' thread            INTEGER NOT NULL,         ' +
       ' caller            INTEGER NULL,             ' +
       ' func              INTEGER NOT NULL,         ' +
       ' start             REAL    NOT NULL,         ' +
       ' end_1             REAL    NOT NULL,         ' +
       ' end_2             REAL    NOT NULL,         ' +
       ' level             INTEGER NOT NULL,         ' +
       ' seconds_self      REAL NOT NULL,            ' +
       ' seconds_total     REAL NOT NULL,            ' +
       ' seconds_dbg_total REAL NOT NULL)'
      ));

    var ac_stmt = null;

    this._ensureNoTransaction();

    try {
      // Immediate so we know our computed IDs are correct
      this._safeSQL('BEGIN IMMEDIATE TRANSACTION');

      // Find the next available ID
      var starting_callno = this._getMaxCallID() + 1;
      var callno = starting_callno;

      try {
        var TYPE_CALL = Ci.IHRProfilerEntry.TYPE_CALL;
        var TYPE_RETURN = Ci.IHRProfilerEntry.TYPE_RETURN;
    
        ac_stmt = safe_create_wrapped_statement(
          this._db,
          ('INSERT INTO temp.unindexed_calls                                ' +
           '       (id,   thread, caller, func, start, end_1, end_2, level, ' +
           '          seconds_self, seconds_total, seconds_dbg_total)       ' +
           'VALUES (?1,   ?2,     ?3,     ?4,   ?5,    ?6,    ?7,    ?8,    ' +
           '          ?9,           ?10,            ?11)                    '));
        
        var acp = ac_stmt.params;
        
        // Cache function IDs so we don't need to hit the database
        var func_cache = {};
        
        if(main_thread_only) {
          var pending = [];
        } else {
          var pending_calls_by_thread = {};
        }

        var num_processed = 0;

        var metadata = {};
        this._metaDataNames.forEach(function(n) metadata[n] = entry[n]);
        metadata.version = 1;
        this._setMetaData(metadata);

        while(!entry.atEOF) {
          var thread = entry.thread;

          if(!main_thread_only) {
            if(thread in pending_calls_by_thread) {
              var pending = pending_calls_by_thread[entry.thread];
            } else {
              var pending = pending_calls_by_thread[entry.thread] = [];
            }
          }
          
          if(entry.type === TYPE_CALL) {
            // pending call record is
            //  0: id of the call
            //  1: time call started
            //  2: accumulated time for all children (excluding dbg time)
            //  3: accumulated debugger time for all children
            //  4: callee (if not commented out)
            pending.push([++callno, entry.startValue,
                          0, 0 /*CALLEE: , entry.callee */]);
          } else if(entry.type === TYPE_RETURN) {
            if(pending.length) {
              var call_callno, call_time;
              var total_child_time, total_child_dbg_time;
              /*CALLEE: var callee; */
              [call_callno, call_time, total_child_time,
               total_child_dbg_time /*CALLEE: , callee*/]
                = pending.pop();

              /* CALLEE:
              if(callee !== entry.callee) {
                dump('MISMATCHED CALLEE');
              }
              */

              if(pending.length) {
                var caller_pending_ent = pending[pending.length - 1];
              } else {
                var caller_pending_ent = null;
              }
              
              var name = entry.name;
              var filename = entry.filename;
              var lineno = entry.lineno;
              
              // Try to find function in cache
              var hashkey = name + '##' + filename + '##' + lineno;
              if(hashkey in func_cache) {
                var func = func_cache[hashkey];
              } else {
                var func = func_cache[hashkey] = this._findOrCreateFunction(
                  name, filename, lineno);
              }

              var end_1 = entry.startValue;
              var end_2 = entry.endValue;

              var total_time = end_1 - call_time - total_child_dbg_time;
              var total_dbg_time = end_2 - end_1 + total_child_dbg_time;

              acp[0]  = call_callno;
              acp[1]  = thread;
              acp[2]  = caller_pending_ent ? caller_pending_ent[0] : null;
              acp[3]  = func;
              acp[4]  = call_time;
              acp[5]  = end_1;
              acp[6]  = end_2;
              acp[7]  = pending.length;
              acp[8]  = total_time - total_child_time;
              acp[9]  = total_time;
              acp[10] = total_dbg_time;
              
              try {
                ac_stmt.execute();
              } catch(ex) {
                ac_stmt.reset();
                report_sql_error(this._db, 'adding call');
              }

              if(caller_pending_ent) {
                caller_pending_ent[2] += total_time;
                caller_pending_ent[3] += total_dbg_time;
              }
              
            } else {
              dump('ignoring orphan return entry\n');
            }
          }
          
          entry.next();
          ++num_processed;

          if(num_processed > chunksize) {
            yield true;
            num_processed = 0;
          }
        }

        this._safeSQL(
          ('INSERT INTO calls SELECT * FROM temp.unindexed_calls'));

      } catch(ex) {
        this._safeSQL('ROLLBACK');
        throw ex;
      }

      this._safeSQL('COMMIT');
    } finally {
      if(ac_stmt) {
        ac_stmt.statement.finalize();
      }
      
      this._safeSQL('DROP TABLE temp.unindexed_calls');
    }
  },

  _profileSelectPart: (
    ('functions.name     AS name,                         ' +
     'functions.filename AS filename,                     ' +
     'functions.lineno   AS lineno,                       ' +
     'functions.id       AS funcid,                       ' +
     'SUM(calls.seconds_total) AS seconds_total,          ' +
     'SUM(calls.seconds_self)  AS seconds_self,           ' +
     'SUM(calls.end_2 - calls.end_1) AS seconds_dbg_self, ' +
     'SUM(calls.seconds_dbg_total) AS seconds_dbg_total,  ' +
     'COUNT(*)           AS calls                         ' )),

  _makeProfileFuncMatchClause: function(func_ids) {
    // SQLite produces better results for a plain equality than a
    // one-element IN clause
    if(func_ids.length === 1) {
      var p = '= ' + func_ids[0];
    } else {
      var p = 'IN (' + func_ids.join(',') + ')';
    }

    return p;
  },

  _getCachedProfileResults: function(tag, sortcol) {
    var sql = ('   SELECT name||"@"||filename||":"||lineno AS fullname, ' +
               '          name, lineno, funcid, seconds_total,          ' +
               '          seconds_self, seconds_dbg_self,               ' +
               '          seconds_dbg_total, calls, filename            ' +
               '     FROM cached_profile_results                        ' +
               '    WHERE query = :tag                                  ' +
               ' ORDER BY ' + sortcol + '                               ' );
    
    var stmt = this._getOrCacheQuery(sql);
    return stmt.queryAllRows({tag: tag});
  },

  /**
    * Enter the results of a profile query into the profile query
    * cache under the given tag. partial_sql is a SELECT query without
    * the 'SELECT' preceding it.
    */
  _cacheProfileResults: function(tag, partial_sql) {
    this._getOrCacheQuery(
      ('DELETE FROM cached_profile_results ' +
       '      WHERE query = :tag           ' ))
      .execute({tag: tag});

    this._safeSQL('INSERT INTO cached_profile_results' +
                  ' SELECT "' + tag + '", ' + partial_sql);
  },

  _cacheDummyResult: function(tag) {
    this._safeSQL(('INSERT INTO cached_profile_results       ' +
                   ' VALUES ("' + tag + '",                  ' +
                   ' "@none@", "",                           ' +
                   ' 0, 0, 0, 0, 0, 0, 0)                    ' ));
  },

  _isDummyResult: function(results) {
    return results.length === 1 && results[0].name === '@none@';
  },

  /**
    * Get an overall profile for the program
    *
    * @param sortcol: name of column to sort by
    * @return: Array of rows
    */
  getFlatProfile: function(sortcol) {
    this._ensureTransaction();
    
    var results = this._getCachedProfileResults('flat', sortcol);
    if(!results.length) {
      var sql = this._profileSelectPart;
      sql += ('    FROM calls, functions          ' +
              '   WHERE calls.func = functions.id ' +
              'GROUP BY calls.func                ' );
      this._cacheProfileResults('flat', sql);
      results = this._getCachedProfileResults('flat', sortcol);
      if(!results.length) {
        this._cacheDummyResult('flat');
      }
    }

    if(this._isDummyResult(results)) {
      results = [];
    }

    return results;
  },

  /**
    * Get a profile for the callers of the given func (a function ID)
    *
    * @param sortcol: name of result column to sort by
    * @param func_ids: Array of ids of functions being called
    * @return: profile
    */
  getCallersProfile: function(sortcol, func_ids) {
    assert(function() typeof sortcol === 'string');
    assert(function() func_ids);
    
    this._ensureTransaction();

    var tag = 'callers-' + func_ids;
    var results = this._getCachedProfileResults(tag, sortcol);
    
    if(!results.length) {
      var sql = this._profileSelectPart;
      var p = this._makeProfileFuncMatchClause(func_ids);

      // We need the SELECT DISTINCT subquery so that we don't
      // double-count the caller of a function if that function calls
      // the function in func_ids multiple times
      sql += (' FROM      (SELECT DISTINCT xcalls.id AS id         ' +
              '              FROM calls AS xcalls,                 ' +
              '                   calls AS xcalls_callee           ' +
              '             WHERE xcalls_callee.caller = xcalls.id ' +
              '               AND xcalls_callee.func ' + p + '     ' +
              '           ) AS x,                                  ' +
              '           functions, calls                         ' +
              '     WHERE calls.id = x.id                          ' +
              '       AND functions.id = calls.func                ' +
              '  GROUP BY calls.func                               ' );
      this._cacheProfileResults(tag, sql);
      results = this._getCachedProfileResults(tag, sortcol);
      if(!results.length) {
        this._cacheDummyResult(tag);
      }
    }

    if(this._isDummyResult(results)) {
      results = [];
    }

    return results;
  },

  /**
    * Get a profile for the functions called by the given func (a function ID)
    *
    * @param sortcol: name of result column to sort by
    * @param func_ids: Arrays of function ids doing the calling
    * @return: profile
    */
  getCalleesProfile: function(sortcol, func_ids) {
    assert(function() typeof sortcol === 'string');
    assert(function() func_ids);

    this._ensureTransaction();

    var tag = 'callees-' + func_ids;
    var results = this._getCachedProfileResults(tag, sortcol);

    if(!results.length) {
      var sql = this._profileSelectPart;
      var p = this._makeProfileFuncMatchClause(func_ids);
    
      sql += ('    FROM calls, functions, calls AS calls_caller ' +
              '   WHERE calls_caller.func ' + p + '             ' +
              '     AND calls.caller = calls_caller.id          ' +
              '     AND functions.id = calls.func               ' +
              'GROUP BY calls.func                              ' );

      this._cacheProfileResults(tag, sql);
      results = this._getCachedProfileResults(tag, sortcol);
      if(!results.length) {
        this._cacheDummyResult(tag);
      }
    }

    
    if(this._isDummyResult(results)) {
      results = [];
    }

    return results;
  },

  /**
    * Return the row for the flat profile corresponding to the given
    * function id.
    *
    * @param funcid: function ID, number
    * @return: row, as in getFlatProfile
    */
  getOverallFuncInfo: function(funcid) {
    assert(function() typeof funcid === 'number');
    return this._getInfoByID()[funcid];
  },

  /**
    * Get metadata corresponding to the whole call graph.
    *
    * Return a metadata object that has at least the following
    * properties:
    *
    *  version: integer, version of profile data
    *
    *  timeSource: string or null, name of time source
    *
    *  timeSourceIsCounter: boolean or null, whether values
    *                       should be interpreted as nanoseconds or
    *                       an integral counter
    *
    * @return: metadata object
    */
  getMetaData: function() {
    return this._metaData;
  },


}

function hrp_profile() {
  if(!gHrp.profiling) {
    hrp_startProfiling();
  } else {
    hrp_endProfiling();
  }
}

