/*
 * GNUsound - a sound editor for GNOME.
 * Copyright (C) 2005  Pascal Haakmat <pascal@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * A copy of the GNU General Public License can be found in the file
 * LICENSE in the top directory of the source distribution. If not,
 * write to the Free Software * Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 *
 */

#ifndef TIMER_H
#define TIMER_H

#define MAX_SRC 10

#define TIMER_FLAGS_DISABLED     (1 << 0)
#define TIMER_FLAGS_UNREGISTERED (1 << 1)

struct time_source {
    int id;
    char *name;
    int flags;
    long long time;
    long long resolution;
};

struct timer {
        int active_src;

        int nsrc;
        struct time_source src[MAX_SRC];
};

/**
 * Initializes global data structures for the timer subsystem and
 * registers a default timer source "Internal" with id 0.
 *
 * @return 0 on success or negative error code.
 */
int 
timer_init();

/**
 * Registers a named time source and returns an integer identifier.
 * Newly registered time sources are disabled by default. The integer
 * identifiers are never reused, even after a call to 
 * timer_unregister_source(). 
 *
 * @param name The source name.
 * @return 0 or negative error code:
 * -EEXIST if a time source by the given name already exists
 * -ENOMEM if there is no space for more time sources
 */
int 
timer_register_source(const char *name);

/**
 * Unregisters a previously registered time source.
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return 0 or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 */
int 
timer_unregister_source(int id);

/**
 * Returns a struct containing info about the next available time source
 * relative to the given integer identifier.
 * All available time sources can be queried by first passing the
 * integer argument -1 and subsequently passing the value of the id field
 * of the returned structures until the function returns NULL.
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return Pointer at time source structure or NULL.
 */
const struct time_source *
timer_get_next_source(int id);

#define timer_get_source(id) timer_get_next_source((id) - 1)

/**
 * Enables a time source, meaning the time source is available for use.
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return 0 or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered time source.
 */
int
timer_enable(int id);

/**
 * Disables a time source, meaning it can no longer be used.
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return 0 or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered time source.
 */
int
timer_disable(int id);

/**
 * Sets the resolution for the time source specified by the integer
 * identifier to the specified amount. 
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @param nsec The resolution in nanoseconds.
 * @return 0 or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered time source. 
 */
int 
timer_set_resolution(int id, 
                     long long nsec);

/**
 * Returns the resolution for the time source specified by the integer
 * identifier.
 *
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return Nanosecond resolution or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered time source. 
 */
long long 
timer_get_resolution(int id);

/**
 * Returns a timer object supporting all the sources registered so far.
 *
 * @return Timer object or NULL on error.
 */
struct timer *
timer_obtain();

/**
 * Discards a timer.
 *
 * @param tmr Timer object obtained via timer_obtain().
 */
void 
timer_discard(struct timer *tmr);

/**
 * Activates a time source. The time source specified by the
 * integer identifier will be used to determine the time. 
 * This deactivates the currently activated time 
 * source, if any. If the id argument is -1, then the currently
 * active time source is deactivated (time stops).
 *
 * @param tmr Timer object obtained via timer_obtain().
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return 0 on success or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered or disabled 
 * time source. 
 */
int 
timer_activate(struct timer *tmr, 
               int id);

/**
 * Sets the time according to the time source specified by the integer
 * identifier.
 *
 * @param tmr Timer object obtained via timer_obtain().
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @param nsec Nanosecond timestamp specifying an absolute time.
 * @return 0 on success or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered or disabled 
 * time source. 
 */
int 
timer_set_time(struct timer *tmr, 
               int id, 
               long long nsec);

/**
 * Adjusts the time for the time source specified by the
 * integer identifier by the specified amount.
 *
 * @param tmr Timer object obtained via timer_obtain().
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @param nsec Nanosecond adjustment.
 * @return 0 on success or a negative error code:
 * @return 0 on success or negative error code:
 * -EINVAL the integer identifier refers to a non-existant time source.
 * -EPERM the integer identifier refers to an unregistered or disabled 
 * time source. 
 */
int 
timer_adjust_time(struct timer *tmr, 
                  int id, 
                  long long nsec);

/**
 * Returns the current time according to the given time source 
 * in nanoseconds.
 *
 * @param tmr Timer object obtained via timer_obtain().
 * @param id Integer identifier of a time source registered using 
 * timer_register_source().
 * @return Nanosecond time stamp or a negative error code:
 * -EINVAL the integer identifier refers to a non-existant, disabled or
 * unregistered time source.
 * -EAGAIN if the time has not yet been determined.
 */
long long 
timer_get_time(struct timer *tmr, 
               int id);

/**
 * Returns the integer identifier of the currently active time source.
 *
 * @param tmr Timer object obtained via timer_obtain().
 * @return Integer identifier or a negative error code:
 * -ENOENT if no time source is currently active.
 */
int 
timer_get_active_source(struct timer *tmr);

#endif /* TIMER_H */
