/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2019 the original author or authors.
 */
package org.assertj.core.internal.strings;

import static org.assertj.core.api.Assertions.assertThatExceptionOfType;
import static org.assertj.core.error.ShouldNotBeBlank.shouldNotBeBlank;
import static org.assertj.core.test.TestData.someInfo;

import java.util.stream.Stream;

import org.assertj.core.internal.StringsBaseTest;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class Strings_assertNotBlank_Test extends StringsBaseTest {

  public static Stream<String> notBlank() {
    return Stream.of("a",
                     " bc ",
                     "\u00A0", // non-breaking space
                     "\u2007", // non-breaking space
                     "\u202F"); // non-breaking space
  }

  @ParameterizedTest
  @MethodSource("notBlank")
  public void should_pass_if_string_is_not_blank(String actual) {
    strings.assertNotBlank(someInfo(), actual);
  }

  public static Stream<String> blank() {
    return Stream.of(null,
                     "",
                     " ",
                     "\u005Ct", // tab
                     "\u005Cn", // line feed
                     "\u005Cr", // carriage return
                     " \u005Cn\u005Cr  "); // non-breaking space
  }

  @ParameterizedTest
  @MethodSource("blank")
  public void should_fail_if_string_is_blank(String actual) {
    assertThatExceptionOfType(AssertionError.class).isThrownBy(() -> strings.assertNotBlank(someInfo(), actual))
                                                   .withMessage(shouldNotBeBlank(actual).create());
  }
}
