/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QObject>

#include "src/identifiers/account_id_db.h"
#include "src/identifiers/message_id.h"
#include "src/worker/task.h"

/*!
 * @brief Restamps message and delivery info in database.
 */
class TaskArchiveIsdsDocument : public QObject, public Task {
	Q_OBJECT

public:
	/*!
	 * @brief Return state describing what happened.
	 */
	enum Result {
		AID_SUCCESS, /*!< Operation was successful. */
		AID_ISDS_ERROR, /*!< Error communicating with ISDS. */
		AID_DB_RD_ERR, /*!< Error reading from database. */
		AID_DB_INS_ERR, /*!< Error inserting into database. */
		AID_ABORTED, /*!< Operation has been aborted. */
		AID_ERR /*!< Other error. */
	};

	explicit TaskArchiveIsdsDocument(const AcntIdDb &acntIdDb,
	    const QString &transactId, const MsgId &msgId,
	    enum MessageDirection direction);

	/*!
	 * @brief Performs actual restamping.
	 */
	virtual
	void run(void) Q_DECL_OVERRIDE;

	/*!
	 * @brief Indicate to the task to quit.
	 */
	void requestQuit(void);

	enum Result m_result; /*!< Return state. */
	QString m_isdsError; /*!< Error description. */
	QString m_isdsLongError; /*!< Long error description. */

	QDateTime m_newExpirationTime; /*!< New timestamp expiration time. */

private Q_SLOTS:
	/*!
	 * @brief Watch ISDS session communication.
	 *
	 * @note Emits uploadProgress() when data uploaded.
	 *
	 * @param[in] uploadTotal Expected total upload,.
	 * @param[in] uploadCurrent Cumulative current upload progress.
	 * @param[in] downloadTotal Expected total download.
	 * @param[in] downloadCurrent Cumulative current download progress.
	 */
	void watchProgress(qint64 uploadTotal, qint64 uploadCurrent,
	    qint64 downloadTotal, qint64 downloadCurrent);

private:
	static
	enum Result archiveMessage(const AcntIdDb &acntIdDb,
	    const MsgId &msgId, enum MessageDirection msgDirect,
	    QDateTime &newExpirationTime,
	    QString &error, QString &longError, volatile bool *reqQuit);

	const AcntIdDb m_acntIdDb; /*!< Account descriptor. */
	const QString m_transactId; /*!< Unique transaction identifier. */
	const MsgId m_msgId; /*!< Message identifier. */
	const enum MessageDirection m_direction; /*!< Message direction. */

	volatile bool m_reqQuit; /*!< Externally requested to quit. */
};

/*!
 * @brief Restamps ZFO from file.
 */
class TaskArchiveIsdsDocumentFile : public QObject, public Task {
	Q_OBJECT

public:
	/*!
	 * @brief Return state describing what happened.
	 */
	enum Result {
		AIDF_SUCCESS, /*!< Operation was successful. */
		AIDF_ISDS_ERROR, /*!< Error communicating with ISDS. */
		AIDF_FILE_RD_ERR, /*!< Error reading file. */
		AIDF_FILE_WR_ERR, /*!< Error writing file. */
		AIDF_ABORTED, /*!< Operation has been aborted. */
		AIDF_ERR /*!< Other error. */
	};

	explicit TaskArchiveIsdsDocumentFile(const AcntIdDb &acntIdDb,
	    const QString &transactId, const QString &srcFile,
	    const QString &tgtFile);

	/*!
	 * @brief Performs actual restamping.
	 */
	virtual
	void run(void) Q_DECL_OVERRIDE;

	/*!
	 * @brief Indicate to the task to quit.
	 */
	void requestQuit(void);

	enum Result m_result; /*!< Return state. */
	QString m_isdsError; /*!< Error description. */
	QString m_isdsLongError; /*!< Long error description. */

	QDateTime m_newExpirationTime; /*!< New timestamp expiration time. */

private Q_SLOTS:
	/*!
	 * @brief Watch ISDS session communication.
	 *
	 * @note Emits uploadProgress() when data uploaded.
	 *
	 * @param[in] uploadTotal Expected total upload,.
	 * @param[in] uploadCurrent Cumulative current upload progress.
	 * @param[in] downloadTotal Expected total download.
	 * @param[in] downloadCurrent Cumulative current download progress.
	 */
	void watchProgress(qint64 uploadTotal, qint64 uploadCurrent,
	    qint64 downloadTotal, qint64 downloadCurrent);

private:
	enum Result archiveFile(const AcntIdDb &acntIdDb,
	    const QString &srcFile, const QString &tgtFile,
	    QDateTime &newExpirationTime,
	    QString &error, QString &longError, volatile bool *reqQuit);

	const AcntIdDb m_acntIdDb; /*!< Account descriptor. */
	const QString m_transactId; /*!< Unique transaction identifier. */
	const QString m_srcFile; /*!< Source file path. */
	const QString m_tgtFile; /*!< Target file path. */

	volatile bool m_reqQuit; /*!< Externally requested to quit. */
};
