/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.uima.fit.pipeline;

import org.apache.uima.UIMAException;
import org.apache.uima.analysis_engine.AnalysisEngineDescription;
import org.apache.uima.collection.CollectionReaderDescription;
import org.apache.uima.fit.internal.ResourceManagerFactory;
import org.apache.uima.jcas.JCas;
import org.apache.uima.resource.ResourceManager;

/**
 * <p>
 * A class implementing iteration over a the documents of a collection. Each element in the Iterable
 * is a JCas containing a single document. The documents have been loaded by the CollectionReader
 * and processed by the AnalysisEngine (if any).
 * </p>
 * <p>
 * <b>NOTE:</b>If any exception is generated while using the iterator generated by
 * {@link #iterator()}, an unchecked {@link IllegalStateException} is thrown!
 * </p>
 * <p>
 * External resources can be shared between the reader and the analysis engines.
 * </p>
 */
public class JCasIterable implements Iterable<JCas> {

  private final ResourceManager resMgr;
  
  private final CollectionReaderDescription reader;

  private final AnalysisEngineDescription[] engines;

  /**
   * Iterate over the documents loaded by the collection reader, running the analysis engines on each
   * one before yielding them. When created with this constructor, analysis engines by default
   * receive a collectionProcessComplete call when all documents have been read from the reader and
   * all components get destroyed.
   * 
   * @param aReader
   *          The collection reader for loading documents.
   * @param aEngines
   *          The analysis engines for processing documents.
   */
  public JCasIterable(final CollectionReaderDescription aReader,
          final AnalysisEngineDescription... aEngines) {
    this(null, aReader, aEngines);
  }

  /**
   * Iterate over the documents loaded by the collection reader, running the analysis engines on each
   * one before yielding them. When created with this constructor, analysis engines by default
   * receive a collectionProcessComplete call when all documents have been read from the reader and
   * all components get destroyed.
   * 
   * @param aResMgr
   *          The {@link ResourceManager} used to create the components and the JCas. If this
   *          parameter is {@code null} then {@link ResourceManagerFactory#newResourceManager()}
   *          will be used to obtain a resource manager. If a new resource manager was internally
   *          created, it is destroyed at the end of the pipeline.
   * @param aReader
   *          The collection reader for loading documents.
   * @param aEngines
   *          The analysis engines for processing documents.
   */
  public JCasIterable(final ResourceManager aResMgr, final CollectionReaderDescription aReader,
          final AnalysisEngineDescription... aEngines) {
    resMgr = aResMgr;
    reader = aReader;
    engines = aEngines;
  }

  @Override
  public JCasIterator iterator() {
    try {
      return new JCasIterator(resMgr, reader, engines);
    } catch (UIMAException e) {
      throw new IllegalStateException(e);
    }
  }
}
