"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const of_1 = require("rxjs/observable/of");
const throw_1 = require("rxjs/observable/throw");
const mergeMap_1 = require("rxjs/operators/mergeMap");
const src_1 = require("../src");
/**
 * An EngineHost that support multiple hosts in a fallback configuration. If a host does not
 * have a collection/schematics, use the following host before giving up.
 */
class FallbackEngineHost {
    constructor() {
        this._hosts = [];
    }
    addHost(host) {
        this._hosts.push(host);
    }
    createCollectionDescription(name) {
        for (const host of this._hosts) {
            try {
                const description = host.createCollectionDescription(name);
                return { name, host, description };
            }
            catch (_) {
            }
        }
        throw new src_1.UnknownCollectionException(name);
    }
    createSchematicDescription(name, collection) {
        const description = collection.host.createSchematicDescription(name, collection.description);
        if (!description) {
            return null;
        }
        return { name, collection, description };
    }
    getSchematicRuleFactory(schematic, collection) {
        return collection.host.getSchematicRuleFactory(schematic.description, collection.description);
    }
    createSourceFromUrl(url, context) {
        return context.schematic.collection.description.host.createSourceFromUrl(url, context);
    }
    transformOptions(schematic, options) {
        return (of_1.of(options)
            .pipe(...this._hosts.map(host => mergeMap_1.mergeMap(opt => host.transformOptions(schematic, opt)))));
    }
    /**
     * @deprecated Use `listSchematicNames`.
     */
    listSchematics(collection) {
        return this.listSchematicNames(collection.description);
    }
    listSchematicNames(collection) {
        const allNames = new Set();
        this._hosts.forEach(host => {
            try {
                host.listSchematicNames(collection.description).forEach(name => allNames.add(name));
            }
            catch (_) { }
        });
        return [...allNames];
    }
    createTaskExecutor(name) {
        for (const host of this._hosts) {
            if (host.hasTaskExecutor(name)) {
                return host.createTaskExecutor(name);
            }
        }
        return throw_1._throw(new src_1.UnregisteredTaskException(name));
    }
    hasTaskExecutor(name) {
        for (const host of this._hosts) {
            if (host.hasTaskExecutor(name)) {
                return true;
            }
        }
        return false;
    }
}
exports.FallbackEngineHost = FallbackEngineHost;
//# sourceMappingURL=data:application/json;base64,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