/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.client.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.EnumSet;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.ReadOption;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.BlockReader;
import org.apache.hadoop.hdfs.client.impl.BlockReaderUtil;
import org.apache.hadoop.hdfs.client.impl.DfsClientConf;
import org.apache.hadoop.hdfs.client.impl.metrics.BlockReaderIoProvider;
import org.apache.hadoop.hdfs.client.impl.metrics.BlockReaderLocalMetrics;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.CachingStrategy;
import org.apache.hadoop.hdfs.shortcircuit.ClientMmap;
import org.apache.hadoop.hdfs.shortcircuit.ShortCircuitReplica;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class BlockReaderLocal
implements BlockReader {
    static final Logger LOG = LoggerFactory.getLogger(BlockReaderLocal.class);
    private static final DirectBufferPool bufferPool = new DirectBufferPool();
    private static BlockReaderLocalMetrics metrics;
    private static Lock metricsInitializationLock;
    private final BlockReaderIoProvider blockReaderIoProvider;
    private static final Timer TIMER;
    private boolean closed = false;
    private final ShortCircuitReplica replica;
    private final FileChannel dataIn;
    private long dataPos;
    private final FileChannel checksumIn;
    private final DataChecksum checksum;
    private final boolean verifyChecksum;
    private final String filename;
    private final ExtendedBlock block;
    private final int bytesPerChecksum;
    private final int checksumSize;
    private final int maxAllocatedChunks;
    private final boolean zeroReadaheadRequested;
    private final int maxReadaheadLength;
    private ByteBuffer dataBuf;
    private ByteBuffer checksumBuf;
    private StorageType storageType;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private BlockReaderLocal(Builder builder) {
        int maxReadaheadChunks;
        this.replica = builder.replica;
        this.dataIn = this.replica.getDataStream().getChannel();
        this.dataPos = builder.dataPos;
        this.checksumIn = this.replica.getMetaStream().getChannel();
        BlockMetadataHeader header = builder.replica.getMetaHeader();
        this.checksum = header.getChecksum();
        this.verifyChecksum = builder.verifyChecksum && this.checksum.getChecksumType().id != 0;
        this.filename = builder.filename;
        this.block = builder.block;
        this.bytesPerChecksum = this.checksum.getBytesPerChecksum();
        this.checksumSize = this.checksum.getChecksumSize();
        this.maxAllocatedChunks = this.bytesPerChecksum == 0 ? 0 : (builder.bufferSize + this.bytesPerChecksum - 1) / this.bytesPerChecksum;
        int n = maxReadaheadChunks = this.bytesPerChecksum == 0 ? 0 : (Math.min(builder.bufferSize, builder.maxReadahead) + this.bytesPerChecksum - 1) / this.bytesPerChecksum;
        if (maxReadaheadChunks == 0) {
            this.zeroReadaheadRequested = true;
            maxReadaheadChunks = 1;
        } else {
            this.zeroReadaheadRequested = false;
        }
        this.maxReadaheadLength = maxReadaheadChunks * this.bytesPerChecksum;
        this.storageType = builder.storageType;
        if (builder.shortCircuitConf.isScrMetricsEnabled()) {
            metricsInitializationLock.lock();
            try {
                if (metrics == null) {
                    metrics = BlockReaderLocalMetrics.create();
                }
            }
            finally {
                metricsInitializationLock.unlock();
            }
        }
        this.blockReaderIoProvider = new BlockReaderIoProvider(builder.shortCircuitConf, metrics, TIMER);
    }

    private synchronized void createDataBufIfNeeded() {
        if (this.dataBuf == null) {
            this.dataBuf = bufferPool.getBuffer(this.maxAllocatedChunks * this.bytesPerChecksum);
            this.dataBuf.position(0);
            this.dataBuf.limit(0);
        }
    }

    private synchronized void freeDataBufIfExists() {
        if (this.dataBuf != null) {
            this.dataPos -= (long)this.dataBuf.remaining();
            this.dataBuf.clear();
            bufferPool.returnBuffer(this.dataBuf);
            this.dataBuf = null;
        }
    }

    private synchronized void createChecksumBufIfNeeded() {
        if (this.checksumBuf == null) {
            this.checksumBuf = bufferPool.getBuffer(this.maxAllocatedChunks * this.checksumSize);
            this.checksumBuf.position(0);
            this.checksumBuf.limit(0);
        }
    }

    private synchronized void freeChecksumBufIfExists() {
        if (this.checksumBuf != null) {
            this.checksumBuf.clear();
            bufferPool.returnBuffer(this.checksumBuf);
            this.checksumBuf = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized int drainDataBuf(ByteBuffer buf) {
        if (this.dataBuf == null) {
            return -1;
        }
        int oldLimit = this.dataBuf.limit();
        int nRead = Math.min(this.dataBuf.remaining(), buf.remaining());
        if (nRead == 0) {
            return this.dataBuf.remaining() == 0 ? -1 : 0;
        }
        try {
            this.dataBuf.limit(this.dataBuf.position() + nRead);
            buf.put(this.dataBuf);
        }
        finally {
            this.dataBuf.limit(oldLimit);
        }
        return nRead;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized int fillBuffer(ByteBuffer buf, boolean canSkipChecksum) throws IOException {
        int nRead;
        int total = 0;
        long startDataPos = this.dataPos;
        int startBufPos = buf.position();
        while (buf.hasRemaining() && (nRead = this.blockReaderIoProvider.read(this.dataIn, buf, this.dataPos)) >= 0) {
            this.dataPos += (long)nRead;
            total += nRead;
        }
        if (canSkipChecksum) {
            this.freeChecksumBufIfExists();
            return total;
        }
        if (total > 0) {
            try {
                buf.limit(buf.position());
                buf.position(startBufPos);
                this.createChecksumBufIfNeeded();
                int checksumsNeeded = (total + this.bytesPerChecksum - 1) / this.bytesPerChecksum;
                this.checksumBuf.clear();
                this.checksumBuf.limit(checksumsNeeded * this.checksumSize);
                long checksumPos = (long)BlockMetadataHeader.getHeaderSize() + startDataPos / (long)this.bytesPerChecksum * (long)this.checksumSize;
                while (this.checksumBuf.hasRemaining()) {
                    int nRead2 = this.checksumIn.read(this.checksumBuf, checksumPos);
                    if (nRead2 < 0) {
                        throw new IOException("Got unexpected checksum file EOF at " + checksumPos + ", block file position " + startDataPos + " for block " + this.block + " of file " + this.filename);
                    }
                    checksumPos += (long)nRead2;
                }
                this.checksumBuf.flip();
                this.checksum.verifyChunkedSums(buf, this.checksumBuf, this.filename, startDataPos);
            }
            finally {
                buf.position(buf.limit());
            }
        }
        return total;
    }

    private boolean createNoChecksumContext() {
        return !this.verifyChecksum || this.storageType != null && this.storageType.isTransient() || this.replica.addNoChecksumAnchor();
    }

    private void releaseNoChecksumContext() {
        if (this.verifyChecksum && (this.storageType == null || !this.storageType.isTransient())) {
            this.replica.removeNoChecksumAnchor();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized int read(ByteBuffer buf) throws IOException {
        boolean canSkipChecksum = this.createNoChecksumContext();
        try {
            int nRead;
            String traceFormatStr = "read(buf.remaining={}, block={}, filename={}, canSkipChecksum={})";
            LOG.trace(traceFormatStr + ": starting", new Object[]{buf.remaining(), this.block, this.filename, canSkipChecksum});
            try {
                nRead = canSkipChecksum && this.zeroReadaheadRequested ? this.readWithoutBounceBuffer(buf) : this.readWithBounceBuffer(buf, canSkipChecksum);
            }
            catch (IOException e) {
                LOG.trace(traceFormatStr + ": I/O error", new Object[]{buf.remaining(), this.block, this.filename, canSkipChecksum, e});
                throw e;
            }
            LOG.trace(traceFormatStr + ": returning {}", new Object[]{buf.remaining(), this.block, this.filename, canSkipChecksum, nRead});
            int n = nRead;
            return n;
        }
        finally {
            if (canSkipChecksum) {
                this.releaseNoChecksumContext();
            }
        }
    }

    private synchronized int readWithoutBounceBuffer(ByteBuffer buf) throws IOException {
        int nRead;
        this.freeDataBufIfExists();
        this.freeChecksumBufIfExists();
        int total = 0;
        while (buf.hasRemaining() && (nRead = this.blockReaderIoProvider.read(this.dataIn, buf, this.dataPos)) > 0) {
            this.dataPos += (long)nRead;
            total += nRead;
        }
        return total == 0 && this.dataPos == this.dataIn.size() ? -1 : total;
    }

    private synchronized boolean fillDataBuf(boolean canSkipChecksum) throws IOException {
        this.createDataBufIfNeeded();
        int slop = (int)(this.dataPos % (long)this.bytesPerChecksum);
        long oldDataPos = this.dataPos;
        this.dataBuf.limit(this.maxReadaheadLength);
        if (canSkipChecksum) {
            this.dataBuf.position(slop);
            this.fillBuffer(this.dataBuf, true);
        } else {
            this.dataPos -= (long)slop;
            this.dataBuf.position(0);
            this.fillBuffer(this.dataBuf, false);
        }
        this.dataBuf.limit(this.dataBuf.position());
        this.dataBuf.position(Math.min(this.dataBuf.position(), slop));
        LOG.trace("loaded {} bytes into bounce buffer from offset {} of {}", new Object[]{this.dataBuf.remaining(), oldDataPos, this.block});
        return this.dataBuf.limit() != this.maxReadaheadLength;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized int readWithBounceBuffer(ByteBuffer buf, boolean canSkipChecksum) throws IOException {
        int total = 0;
        int bb = this.drainDataBuf(buf);
        if (bb >= 0) {
            total += bb;
            if (buf.remaining() == 0) {
                return total;
            }
        }
        boolean eof = true;
        boolean done = false;
        do {
            if (buf.isDirect() && buf.remaining() >= this.maxReadaheadLength && this.dataPos % (long)this.bytesPerChecksum == 0L) {
                int nRead;
                int oldLimit = buf.limit();
                try {
                    buf.limit(buf.position() + this.maxReadaheadLength);
                    nRead = this.fillBuffer(buf, canSkipChecksum);
                }
                finally {
                    buf.limit(oldLimit);
                }
                if (nRead < this.maxReadaheadLength) {
                    done = true;
                }
                if (nRead > 0) {
                    eof = false;
                }
                total += nRead;
                continue;
            }
            if (this.fillDataBuf(canSkipChecksum)) {
                done = true;
            }
            if ((bb = this.drainDataBuf(buf)) < 0) continue;
            eof = false;
            total += bb;
        } while (!done && buf.remaining() > 0);
        return eof && total == 0 ? -1 : total;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized int read(byte[] arr, int off, int len) throws IOException {
        int nRead;
        boolean canSkipChecksum = this.createNoChecksumContext();
        try {
            String traceFormatStr = "read(arr.length={}, off={}, len={}, filename={}, block={}, canSkipChecksum={})";
            LOG.trace("read(arr.length={}, off={}, len={}, filename={}, block={}, canSkipChecksum={}): starting", new Object[]{arr.length, off, len, this.filename, this.block, canSkipChecksum});
            try {
                nRead = canSkipChecksum && this.zeroReadaheadRequested ? this.readWithoutBounceBuffer(arr, off, len) : this.readWithBounceBuffer(arr, off, len, canSkipChecksum);
            }
            catch (IOException e) {
                LOG.trace("read(arr.length={}, off={}, len={}, filename={}, block={}, canSkipChecksum={}): I/O error", new Object[]{arr.length, off, len, this.filename, this.block, canSkipChecksum, e});
                throw e;
            }
            LOG.trace("read(arr.length={}, off={}, len={}, filename={}, block={}, canSkipChecksum={}): returning {}", new Object[]{arr.length, off, len, this.filename, this.block, canSkipChecksum, nRead});
        }
        finally {
            if (canSkipChecksum) {
                this.releaseNoChecksumContext();
            }
        }
        return nRead;
    }

    private synchronized int readWithoutBounceBuffer(byte[] arr, int off, int len) throws IOException {
        this.freeDataBufIfExists();
        this.freeChecksumBufIfExists();
        int nRead = this.blockReaderIoProvider.read(this.dataIn, ByteBuffer.wrap(arr, off, len), this.dataPos);
        if (nRead > 0) {
            this.dataPos += (long)nRead;
        } else if (nRead == 0 && this.dataPos == this.dataIn.size()) {
            return -1;
        }
        return nRead;
    }

    private synchronized int readWithBounceBuffer(byte[] arr, int off, int len, boolean canSkipChecksum) throws IOException {
        this.createDataBufIfNeeded();
        if (!this.dataBuf.hasRemaining()) {
            this.dataBuf.position(0);
            this.dataBuf.limit(this.maxReadaheadLength);
            this.fillDataBuf(canSkipChecksum);
        }
        if (this.dataBuf.remaining() == 0) {
            return -1;
        }
        int toRead = Math.min(this.dataBuf.remaining(), len);
        this.dataBuf.get(arr, off, toRead);
        return toRead;
    }

    @Override
    public synchronized long skip(long n) throws IOException {
        int discardedFromBuf = 0;
        long remaining = n;
        if (this.dataBuf != null && this.dataBuf.hasRemaining()) {
            discardedFromBuf = (int)Math.min((long)this.dataBuf.remaining(), n);
            this.dataBuf.position(this.dataBuf.position() + discardedFromBuf);
            remaining -= (long)discardedFromBuf;
        }
        LOG.trace("skip(n={}, block={}, filename={}): discarded {} bytes from dataBuf and advanced dataPos by {}", new Object[]{n, this.block, this.filename, discardedFromBuf, remaining});
        this.dataPos += remaining;
        return n;
    }

    @Override
    public int available() {
        return Integer.MAX_VALUE;
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        LOG.trace("close(filename={}, block={})", (Object)this.filename, (Object)this.block);
        this.replica.unref();
        this.freeDataBufIfExists();
        this.freeChecksumBufIfExists();
        if (metrics != null) {
            metrics.collectThreadLocalStates();
        }
    }

    @Override
    public synchronized void readFully(byte[] arr, int off, int len) throws IOException {
        BlockReaderUtil.readFully(this, arr, off, len);
    }

    @Override
    public synchronized int readAll(byte[] buf, int off, int len) throws IOException {
        return BlockReaderUtil.readAll(this, buf, off, len);
    }

    @Override
    public boolean isShortCircuit() {
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ClientMmap getClientMmap(EnumSet<ReadOption> opts) {
        boolean anchor;
        boolean bl = anchor = this.verifyChecksum && !opts.contains(ReadOption.SKIP_CHECKSUMS);
        if (anchor && !this.createNoChecksumContext()) {
            LOG.trace("can't get an mmap for {} of {} since SKIP_CHECKSUMS was not given, we aren't skipping checksums, and the block is not mlocked.", (Object)this.block, (Object)this.filename);
            return null;
        }
        ClientMmap clientMmap = null;
        try {
            clientMmap = this.replica.getOrCreateClientMmap(anchor);
        }
        finally {
            if (clientMmap == null && anchor) {
                this.releaseNoChecksumContext();
            }
        }
        return clientMmap;
    }

    @VisibleForTesting
    boolean getVerifyChecksum() {
        return this.verifyChecksum;
    }

    @VisibleForTesting
    int getMaxReadaheadLength() {
        return this.maxReadaheadLength;
    }

    @VisibleForTesting
    void forceAnchorable() {
        this.replica.getSlot().makeAnchorable();
    }

    @VisibleForTesting
    void forceUnanchorable() {
        this.replica.getSlot().makeUnanchorable();
    }

    @Override
    public DataChecksum getDataChecksum() {
        return this.checksum;
    }

    @Override
    public int getNetworkDistance() {
        return 0;
    }

    static {
        metricsInitializationLock = new ReentrantLock();
        TIMER = new Timer();
    }

    public static class Builder {
        private final int bufferSize;
        private boolean verifyChecksum;
        private int maxReadahead;
        private String filename;
        private ShortCircuitReplica replica;
        private long dataPos;
        private ExtendedBlock block;
        private StorageType storageType;
        private DfsClientConf.ShortCircuitConf shortCircuitConf;

        public Builder(DfsClientConf.ShortCircuitConf conf) {
            this.shortCircuitConf = conf;
            this.maxReadahead = Integer.MAX_VALUE;
            this.verifyChecksum = !conf.isSkipShortCircuitChecksums();
            this.bufferSize = conf.getShortCircuitBufferSize();
        }

        public Builder setVerifyChecksum(boolean verifyChecksum) {
            this.verifyChecksum = verifyChecksum;
            return this;
        }

        public Builder setCachingStrategy(CachingStrategy cachingStrategy) {
            long readahead = cachingStrategy.getReadahead() != null ? cachingStrategy.getReadahead() : 0x400000L;
            this.maxReadahead = (int)Math.min(Integer.MAX_VALUE, readahead);
            return this;
        }

        public Builder setFilename(String filename) {
            this.filename = filename;
            return this;
        }

        public Builder setShortCircuitReplica(ShortCircuitReplica replica) {
            this.replica = replica;
            return this;
        }

        public Builder setStartOffset(long startOffset) {
            this.dataPos = Math.max(0L, startOffset);
            return this;
        }

        public Builder setBlock(ExtendedBlock block) {
            this.block = block;
            return this;
        }

        public Builder setStorageType(StorageType storageType) {
            this.storageType = storageType;
            return this;
        }

        public BlockReaderLocal build() {
            Preconditions.checkNotNull((Object)this.replica);
            return new BlockReaderLocal(this);
        }
    }
}

