/*
 * Decompiled with CFR 0.152.
 */
package org.apache.baremaps.workflow;

import com.google.common.graph.Graph;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.Graphs;
import com.google.common.graph.ImmutableGraph;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import org.apache.baremaps.workflow.Step;
import org.apache.baremaps.workflow.Task;
import org.apache.baremaps.workflow.Workflow;
import org.apache.baremaps.workflow.WorkflowContext;
import org.apache.baremaps.workflow.WorkflowException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkflowExecutor
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(WorkflowExecutor.class);
    private final ExecutorService executorService;
    private final WorkflowContext context;
    private final Map<String, Step> steps;
    private final Map<String, CompletableFuture<Void>> futures;
    private final Graph<String> graph;
    private final List<StepMeasure> stepMeasures;

    public WorkflowExecutor(Workflow workflow) {
        this(workflow, Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors()));
    }

    public WorkflowExecutor(Workflow workflow, ExecutorService executorService) {
        this.executorService = executorService;
        this.context = new WorkflowContext();
        this.steps = workflow.getSteps().stream().collect(Collectors.toMap(step -> step.getId(), step -> step));
        this.futures = new ConcurrentSkipListMap<String, CompletableFuture<Void>>();
        this.stepMeasures = new CopyOnWriteArrayList<StepMeasure>();
        ImmutableGraph.Builder graphBuilder = GraphBuilder.directed().immutable();
        for (String id : this.steps.keySet()) {
            graphBuilder.addNode((Object)id);
        }
        for (Step step2 : this.steps.values()) {
            if (step2.getNeeds() == null) continue;
            for (String stepNeeded : step2.getNeeds()) {
                graphBuilder.putEdge((Object)stepNeeded, (Object)step2.getId());
            }
        }
        this.graph = graphBuilder.build();
        if (Graphs.hasCycle(this.graph)) {
            throw new WorkflowException("The workflow must be a directed acyclic graph");
        }
        logger.info("Workflow graph: {}", this.graph);
    }

    public CompletableFuture<Void> execute() {
        CompletableFuture[] endSteps = (CompletableFuture[])this.graph.nodes().stream().filter(this::isEndStep).map(this::getFutureStep).toArray(CompletableFuture[]::new);
        CompletionStage future = CompletableFuture.allOf(endSteps).thenRun(this::logStepMeasures);
        return future;
    }

    private CompletableFuture<Void> getFutureStep(String step) {
        return this.futures.computeIfAbsent(step, this::createFutureStep);
    }

    private CompletableFuture<Void> createFutureStep(String stepId) {
        CompletionStage<Void> future = this.getPreviousFutureStep(stepId);
        ArrayList<TaskMeasure> measures = new ArrayList<TaskMeasure>();
        Step step = this.steps.get(stepId);
        if (step == null) {
            logger.warn("Step {} does not exist and will be skipped", (Object)stepId);
            return future;
        }
        List<Task> tasks = step.getTasks();
        for (Task task : tasks) {
            future = future.thenRunAsync(() -> {
                try {
                    logger.info("Executing task {} of step {}", (Object)task, (Object)stepId);
                    long start = System.currentTimeMillis();
                    task.execute(this.context);
                    long end = System.currentTimeMillis();
                    TaskMeasure measure = new TaskMeasure(task, start, end);
                    measures.add(measure);
                }
                catch (Exception e) {
                    throw new WorkflowException(e);
                }
            }, this.executorService);
        }
        this.stepMeasures.add(new StepMeasure(step, measures));
        return future;
    }

    private CompletableFuture<Void> getPreviousFutureStep(String stepId) {
        List predecessors = this.graph.predecessors((Object)stepId).stream().toList();
        if (predecessors.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        if (predecessors.size() == 1) {
            return this.getFutureStep((String)predecessors.get(0));
        }
        CompletableFuture[] futurePredecessors = (CompletableFuture[])predecessors.stream().map(this::getFutureStep).toArray(CompletableFuture[]::new);
        return CompletableFuture.allOf(futurePredecessors);
    }

    private void logStepMeasures() {
        logger.info("----------------------------------------");
        long workflowStart = this.stepMeasures.stream().mapToLong(measures -> measures.stepMeasures.stream().mapToLong(measure -> measure.start).min().getAsLong()).min().getAsLong();
        long workflowEnd = this.stepMeasures.stream().mapToLong(measures -> measures.stepMeasures.stream().mapToLong(measure -> measure.end).max().getAsLong()).max().getAsLong();
        Duration workflowDuration = Duration.ofMillis(workflowEnd - workflowStart);
        logger.info("Workflow graph: {}", this.graph);
        logger.info("  Duration: {}", (Object)WorkflowExecutor.formatDuration(workflowDuration));
        for (StepMeasure stepMeasure : this.stepMeasures) {
            long stepStart = stepMeasure.stepMeasures.stream().mapToLong(measure -> measure.start).min().getAsLong();
            long stepEnd = stepMeasure.stepMeasures.stream().mapToLong(measure -> measure.end).max().getAsLong();
            Duration stepDuration = Duration.ofMillis(stepEnd - stepStart);
            logger.info("Step: {}, Duration: {} ms", (Object)stepMeasure.step.getId(), (Object)WorkflowExecutor.formatDuration(stepDuration));
            for (TaskMeasure taskMeasure : stepMeasure.stepMeasures) {
                Duration taskDuration = Duration.ofMillis(taskMeasure.end - taskMeasure.start);
                logger.info("  Task: {}", (Object)taskMeasure.task);
                logger.info("    Duration: {}", (Object)WorkflowExecutor.formatDuration(taskDuration));
            }
        }
        logger.info("----------------------------------------");
    }

    private static String formatDuration(Duration duration) {
        long ms;
        long sec;
        long min;
        long hrs;
        StringBuilder builder = new StringBuilder();
        long days = duration.toDays();
        if (days > 0L) {
            builder.append(days).append(" days ");
        }
        if ((hrs = duration.toHours() - Duration.ofDays(duration.toDays()).toHours()) > 0L) {
            builder.append(hrs).append(" hrs ");
        }
        if ((min = duration.toMinutes() - Duration.ofHours(duration.toHours()).toMinutes()) > 0L) {
            builder.append(min).append(" min ");
        }
        if ((sec = duration.toSeconds() - Duration.ofMinutes(duration.toMinutes()).toSeconds()) > 0L) {
            builder.append(sec).append(" s ");
        }
        if ((ms = duration.toMillis() - Duration.ofSeconds(duration.toSeconds()).toMillis()) > 0L) {
            builder.append(ms).append(" ms ");
        }
        return builder.toString();
    }

    private boolean isEndStep(String stepId) {
        return this.graph.successors((Object)stepId).isEmpty();
    }

    @Override
    public void close() throws Exception {
        this.executorService.shutdown();
    }

    record StepMeasure(Step step, List<TaskMeasure> stepMeasures) {
    }

    record TaskMeasure(Task task, long start, long end) {
    }
}

