/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.binarytuple;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.util.UUID;
import org.apache.ignite3.internal.binarytuple.BinaryTupleCommon;
import org.apache.ignite3.internal.binarytuple.BinaryTupleFormatException;
import org.apache.ignite3.internal.util.ByteUtils;
import org.jetbrains.annotations.Nullable;

public class BinaryTupleBuilder {
    private static final int DEFAULT_BUFFER_SIZE = 1024;
    private int elementIndex = 0;
    private final int numElements;
    private final int entrySize;
    private final int entryBase;
    private final int valueBase;
    private ByteBuffer buffer;
    private CharsetEncoder cachedEncoder;

    public BinaryTupleBuilder(int numElements) {
        this(numElements, -1);
    }

    public BinaryTupleBuilder(int numElements, int totalValueSize) {
        this(numElements, totalValueSize, true);
    }

    public BinaryTupleBuilder(int numElements, int totalValueSize, boolean exactEstimate) {
        this.numElements = numElements;
        this.entryBase = 1;
        this.entrySize = totalValueSize < 0 || !exactEstimate ? 4 : BinaryTupleCommon.flagsToEntrySize(BinaryTupleCommon.valueSizeToFlags(totalValueSize));
        this.valueBase = this.entryBase + this.entrySize * numElements;
        this.allocate(totalValueSize);
    }

    public BinaryTupleBuilder appendNull() {
        return this.proceed();
    }

    public BinaryTupleBuilder appendBoolean(boolean value) {
        this.putByte(ByteUtils.booleanToByte(value));
        return this.proceed();
    }

    public BinaryTupleBuilder appendBoolean(Boolean value) {
        return value == null ? this.appendNull() : this.appendBoolean((boolean)value);
    }

    public BinaryTupleBuilder appendByte(byte value) {
        this.putByte(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendByte(Byte value) {
        return value == null ? this.appendNull() : this.appendByte((byte)value);
    }

    public BinaryTupleBuilder appendShort(short value) {
        if (-128 <= value && value <= 127) {
            return this.appendByte((byte)value);
        }
        this.putShort(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendShort(Short value) {
        return value == null ? this.appendNull() : this.appendShort((short)value);
    }

    public BinaryTupleBuilder appendInt(int value) {
        if (-128 <= value && value <= 127) {
            return this.appendByte((byte)value);
        }
        if (Short.MIN_VALUE <= value && value <= Short.MAX_VALUE) {
            this.putShort((short)value);
        } else {
            this.putInt(value);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendInt(@Nullable Integer value) {
        return value == null ? this.appendNull() : this.appendInt((int)value);
    }

    public BinaryTupleBuilder appendLong(long value) {
        if (-32768L <= value && value <= 32767L) {
            return this.appendShort((short)value);
        }
        if (Integer.MIN_VALUE <= value && value <= Integer.MAX_VALUE) {
            this.putInt((int)value);
        } else {
            this.putLong(value);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendLong(Long value) {
        return value == null ? this.appendNull() : this.appendLong((long)value);
    }

    public BinaryTupleBuilder appendFloat(float value) {
        this.putFloat(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendFloat(Float value) {
        return value == null ? this.appendNull() : this.appendFloat(value.floatValue());
    }

    public BinaryTupleBuilder appendDouble(double value) {
        if (value == (double)((float)value)) {
            return this.appendFloat((float)value);
        }
        this.putDouble(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendDouble(Double value) {
        return value == null ? this.appendNull() : this.appendDouble((double)value);
    }

    public BinaryTupleBuilder appendDecimalNotNull(BigDecimal value, int scale) {
        if ((value = BinaryTupleCommon.shrinkDecimal(value, scale)).scale() > Short.MAX_VALUE) {
            throw new BinaryTupleFormatException("Decimal scale is too large: " + value.scale() + " > 32767");
        }
        if (value.scale() < Short.MIN_VALUE) {
            throw new BinaryTupleFormatException("Decimal scale is too small: " + value.scale() + " < -32768");
        }
        this.putShort((short)value.scale());
        this.putBytes(value.unscaledValue().toByteArray());
        return this.proceed();
    }

    public BinaryTupleBuilder appendDecimal(BigDecimal value, int scale) {
        return value == null ? this.appendNull() : this.appendDecimalNotNull(value, scale);
    }

    public BinaryTupleBuilder appendStringNotNull(String value) {
        try {
            this.putString(value);
        }
        catch (CharacterCodingException e) {
            throw new BinaryTupleFormatException("Failed to encode string in binary tuple builder", (Throwable)e);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendString(@Nullable String value) {
        return value == null ? this.appendNull() : this.appendStringNotNull(value);
    }

    public BinaryTupleBuilder appendBytesNotNull(byte[] value) {
        this.putBytesWithEmptyCheck(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendBytes(byte @Nullable [] value) {
        return value == null ? this.appendNull() : this.appendBytesNotNull(value);
    }

    public BinaryTupleBuilder appendUuidNotNull(UUID value) {
        this.putLong(value.getMostSignificantBits());
        this.putLong(value.getLeastSignificantBits());
        return this.proceed();
    }

    public BinaryTupleBuilder appendUuid(UUID value) {
        return value == null ? this.appendNull() : this.appendUuidNotNull(value);
    }

    public BinaryTupleBuilder appendDateNotNull(LocalDate value) {
        this.putDate(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendDate(LocalDate value) {
        return value == null ? this.appendNull() : this.appendDateNotNull(value);
    }

    public BinaryTupleBuilder appendTimeNotNull(LocalTime value) {
        this.putTime(value);
        return this.proceed();
    }

    public BinaryTupleBuilder appendTime(LocalTime value) {
        return value == null ? this.appendNull() : this.appendTimeNotNull(value);
    }

    public BinaryTupleBuilder appendDateTimeNotNull(LocalDateTime value) {
        this.putDate(value.toLocalDate());
        this.putTime(value.toLocalTime());
        return this.proceed();
    }

    public BinaryTupleBuilder appendDateTime(LocalDateTime value) {
        return value == null ? this.appendNull() : this.appendDateTimeNotNull(value);
    }

    public BinaryTupleBuilder appendTimestampNotNull(Instant value) {
        long seconds = value.getEpochSecond();
        int nanos = value.getNano();
        this.putLong(seconds);
        if (nanos != 0) {
            this.putInt(nanos);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendTimestamp(Instant value) {
        return value == null ? this.appendNull() : this.appendTimestampNotNull(value);
    }

    public BinaryTupleBuilder appendDurationNotNull(Duration value) {
        long seconds = value.getSeconds();
        int nanos = value.getNano();
        this.putLong(seconds);
        if (nanos != 0) {
            this.putInt(nanos);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendDuration(Duration value) {
        return value == null ? this.appendNull() : this.appendDurationNotNull(value);
    }

    public BinaryTupleBuilder appendPeriodNotNull(Period value) {
        int years = value.getYears();
        int months = value.getMonths();
        int days = value.getDays();
        if (-128 <= years && years <= 127 && -128 <= months && months <= 127 && -128 <= days && days <= 127) {
            this.putByte((byte)years);
            this.putByte((byte)months);
            this.putByte((byte)days);
        } else if (Short.MIN_VALUE <= years && years <= Short.MAX_VALUE && Short.MIN_VALUE <= months && months <= Short.MAX_VALUE && Short.MIN_VALUE <= days && days <= Short.MAX_VALUE) {
            this.putShort((short)years);
            this.putShort((short)months);
            this.putShort((short)days);
        } else {
            this.putInt(years);
            this.putInt(months);
            this.putInt(days);
        }
        return this.proceed();
    }

    public BinaryTupleBuilder appendPeriod(Period value) {
        return value == null ? this.appendNull() : this.appendPeriodNotNull(value);
    }

    public BinaryTupleBuilder appendElementBytes(ByteBuffer bytes, int offset, int length) {
        this.putElement(bytes, offset, length);
        return this.proceed();
    }

    public int elementIndex() {
        return this.elementIndex;
    }

    public int numElements() {
        return this.numElements;
    }

    public ByteBuffer build() {
        return this.buildInternal().slice().order(ByteOrder.LITTLE_ENDIAN);
    }

    protected ByteBuffer buildInternal() {
        int offset = 0;
        int valueSize = this.buffer.position() - this.valueBase;
        byte flags = BinaryTupleCommon.valueSizeToFlags(valueSize);
        int desiredEntrySize = BinaryTupleCommon.flagsToEntrySize(flags);
        if (desiredEntrySize != this.entrySize) {
            if (desiredEntrySize > this.entrySize) {
                throw new IllegalStateException("Offset entry overflow in binary tuple builder");
            }
            assert (this.entrySize == 4 || this.entrySize == 2);
            assert (desiredEntrySize == 2 || desiredEntrySize == 1);
            int getIndex = this.valueBase;
            int putIndex = this.valueBase;
            while (getIndex > this.entryBase) {
                putIndex -= desiredEntrySize;
                int value = this.entrySize == 4 ? this.buffer.getInt(getIndex) : Short.toUnsignedInt(this.buffer.getShort(getIndex -= this.entrySize));
                if (desiredEntrySize == 1) {
                    this.buffer.put(putIndex, (byte)value);
                    continue;
                }
                this.buffer.putShort(putIndex, (short)value);
            }
            offset = (this.entrySize - desiredEntrySize) * this.numElements;
        }
        this.buffer.put(offset, flags);
        return this.buffer.flip().position(offset);
    }

    private void putByte(byte value) {
        this.ensure(1);
        this.buffer.put(value);
    }

    private void putShort(short value) {
        this.ensure(2);
        this.buffer.putShort(value);
    }

    private void putInt(int value) {
        this.ensure(4);
        this.buffer.putInt(value);
    }

    private void putLong(long value) {
        this.ensure(8);
        this.buffer.putLong(value);
    }

    private void putFloat(float value) {
        this.ensure(4);
        this.buffer.putFloat(value);
    }

    private void putDouble(double value) {
        this.ensure(8);
        this.buffer.putDouble(value);
    }

    private void putBytes(byte[] bytes) {
        this.ensure(bytes.length);
        this.buffer.put(bytes);
    }

    private void putBytesWithEmptyCheck(byte[] bytes) {
        if (bytes.length == 0 || bytes[0] == -128) {
            this.ensure(bytes.length + 1);
            this.buffer.put((byte)-128);
        } else {
            this.ensure(bytes.length);
        }
        this.buffer.put(bytes);
    }

    private void putString(String value) throws CharacterCodingException {
        if (value.isEmpty()) {
            this.ensure(1);
            this.buffer.put((byte)-128);
            return;
        }
        int begin = this.buffer.position();
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        this.putBytes(bytes);
        if (this.buffer.get(begin) == -128) {
            throw new BinaryTupleFormatException("Failed to encode a string element: resulting payload starts with invalid 0x80 byte");
        }
    }

    private void putDate(LocalDate value) {
        int year = value.getYear();
        int month = value.getMonthValue();
        int day = value.getDayOfMonth();
        int date = year << 9 | month << 5 | day;
        this.putShort((short)date);
        this.putByte((byte)(date >> 16));
    }

    private void putTime(LocalTime value) {
        long hour = value.getHour();
        long minute = value.getMinute();
        long second = value.getSecond();
        long nanos = value.getNano();
        if (nanos % 1000L != 0L) {
            long time = hour << 42 | minute << 36 | second << 30 | nanos;
            this.putInt((int)time);
            this.putShort((short)(time >>> 32));
        } else if (nanos % 1000000L != 0L) {
            long time = hour << 32 | minute << 26 | second << 20 | nanos / 1000L;
            this.putInt((int)time);
            this.putByte((byte)(time >>> 32));
        } else {
            long time = hour << 22 | minute << 16 | second << 10 | nanos / 1000000L;
            this.putInt((int)time);
        }
    }

    private void putElement(ByteBuffer bytes, int offset, int length) {
        assert (bytes.limit() >= offset + length);
        this.ensure(length);
        this.buffer.put(bytes.asReadOnlyBuffer().position(offset).limit(offset + length));
    }

    private BinaryTupleBuilder proceed() {
        assert (this.elementIndex < this.numElements) : "Element index overflow: " + this.elementIndex + " >= " + this.numElements;
        int offset = this.buffer.position() - this.valueBase;
        switch (this.entrySize) {
            case 1: {
                this.buffer.put(this.entryBase + this.elementIndex, (byte)offset);
                break;
            }
            case 2: {
                this.buffer.putShort(this.entryBase + this.elementIndex * 2, (short)offset);
                break;
            }
            case 4: {
                this.buffer.putInt(this.entryBase + this.elementIndex * 4, offset);
                break;
            }
            default: {
                assert (false);
                break;
            }
        }
        ++this.elementIndex;
        return this;
    }

    private void allocate(int totalValueSize) {
        this.buffer = ByteBuffer.allocate(this.estimateBufferCapacity(totalValueSize));
        this.buffer.order(ByteOrder.LITTLE_ENDIAN);
        this.buffer.position(this.valueBase);
    }

    private int estimateBufferCapacity(int totalValueSize) {
        if (totalValueSize < 0) {
            totalValueSize = Integer.max(this.numElements * 8, 1024);
        }
        return this.valueBase + totalValueSize;
    }

    private void ensure(int size) {
        if (this.buffer.remaining() < size) {
            this.grow(size);
        }
    }

    private void grow(int size) {
        int capacity = this.buffer.capacity();
        do {
            if ((capacity *= 2) >= 0) continue;
            throw new BinaryTupleFormatException("Buffer overflow in binary tuple builder");
        } while (capacity - this.buffer.position() < size);
        ByteBuffer newBuffer = ByteBuffer.allocate(capacity);
        newBuffer.order(ByteOrder.LITTLE_ENDIAN);
        newBuffer.put(this.buffer.flip());
        this.buffer = newBuffer;
    }

    private CharsetEncoder encoder() {
        if (this.cachedEncoder == null) {
            this.cachedEncoder = StandardCharsets.UTF_8.newEncoder();
        }
        return this.cachedEncoder;
    }
}

