/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.index;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.ignite.configuration.notifications.ConfigurationNamedListListener;
import org.apache.ignite.configuration.notifications.ConfigurationNotificationEvent;
import org.apache.ignite.internal.index.ColumnCollation;
import org.apache.ignite.internal.index.HashIndex;
import org.apache.ignite.internal.index.Index;
import org.apache.ignite.internal.index.IndexDescriptor;
import org.apache.ignite.internal.index.SortedIndex;
import org.apache.ignite.internal.index.SortedIndexDescriptor;
import org.apache.ignite.internal.index.SortedIndexImpl;
import org.apache.ignite.internal.index.event.IndexEvent;
import org.apache.ignite.internal.index.event.IndexEventParameters;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.manager.Event;
import org.apache.ignite.internal.manager.IgniteComponent;
import org.apache.ignite.internal.manager.Producer;
import org.apache.ignite.internal.schema.BinaryConverter;
import org.apache.ignite.internal.schema.BinaryRow;
import org.apache.ignite.internal.schema.BinaryTuple;
import org.apache.ignite.internal.schema.BinaryTupleSchema;
import org.apache.ignite.internal.schema.Column;
import org.apache.ignite.internal.schema.SchemaDescriptor;
import org.apache.ignite.internal.schema.SchemaManager;
import org.apache.ignite.internal.schema.SchemaRegistry;
import org.apache.ignite.internal.schema.configuration.ExtendedTableConfiguration;
import org.apache.ignite.internal.schema.configuration.TableConfiguration;
import org.apache.ignite.internal.schema.configuration.TablesConfiguration;
import org.apache.ignite.internal.schema.configuration.index.HashIndexChange;
import org.apache.ignite.internal.schema.configuration.index.HashIndexView;
import org.apache.ignite.internal.schema.configuration.index.IndexColumnView;
import org.apache.ignite.internal.schema.configuration.index.SortedIndexView;
import org.apache.ignite.internal.schema.configuration.index.TableIndexChange;
import org.apache.ignite.internal.schema.configuration.index.TableIndexConfiguration;
import org.apache.ignite.internal.schema.configuration.index.TableIndexView;
import org.apache.ignite.internal.table.TableImpl;
import org.apache.ignite.internal.table.distributed.TableManager;
import org.apache.ignite.internal.table.event.TableEvent;
import org.apache.ignite.internal.util.ArrayUtils;
import org.apache.ignite.internal.util.IgniteSpinBusyLock;
import org.apache.ignite.internal.util.StringUtils;
import org.apache.ignite.lang.ErrorGroups;
import org.apache.ignite.lang.IgniteInternalException;
import org.apache.ignite.lang.IndexAlreadyExistsException;
import org.apache.ignite.lang.IndexNotFoundException;
import org.apache.ignite.lang.NodeStoppingException;
import org.apache.ignite.lang.TableNotFoundException;
import org.jetbrains.annotations.NotNull;

public class IndexManager
extends Producer<IndexEvent, IndexEventParameters>
implements IgniteComponent {
    private static final IgniteLogger LOG = Loggers.forClass(IndexManager.class);
    private final TablesConfiguration tablesCfg;
    private final SchemaManager schemaManager;
    private final TableManager tableManager;
    private final IgniteSpinBusyLock busyLock = new IgniteSpinBusyLock();
    private final AtomicBoolean stopGuard = new AtomicBoolean();

    public IndexManager(TablesConfiguration tablesCfg, SchemaManager schemaManager, TableManager tableManager) {
        this.tablesCfg = Objects.requireNonNull(tablesCfg, "tablesCfg");
        this.schemaManager = Objects.requireNonNull(schemaManager, "schemaManager");
        this.tableManager = tableManager;
    }

    public void start() {
        LOG.debug("Index manager is about to start", new Object[0]);
        this.tablesCfg.indexes().listenElements((ConfigurationNamedListListener)new ConfigurationListener());
        this.tableManager.listen((Event)TableEvent.CREATE, (param, ex) -> {
            if (ex != null) {
                return CompletableFuture.completedFuture(false);
            }
            List pkColumns = Arrays.stream(param.table().schemaView().schema().keyColumns().columns()).map(Column::name).collect(Collectors.toList());
            String pkName = param.tableName() + "_PK";
            this.createIndexAsync("PUBLIC", pkName, param.tableName(), false, change -> ((HashIndexChange)change.changeUniq(true).convert(HashIndexChange.class)).changeColumnNames(pkColumns.toArray(ArrayUtils.STRING_EMPTY_ARRAY)));
            return CompletableFuture.completedFuture(false);
        });
        LOG.info("Index manager started", new Object[0]);
    }

    public void stop() throws Exception {
        LOG.debug("Index manager is about to stop", new Object[0]);
        if (!this.stopGuard.compareAndSet(false, true)) {
            LOG.debug("Index manager already stopped", new Object[0]);
            return;
        }
        this.busyLock.block();
        LOG.info("Index manager stopped", new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<Boolean> createIndexAsync(String schemaName, String indexName, String tableName, boolean failIfExists, Consumer<TableIndexChange> indexChange) {
        if (!this.busyLock.enterBusy()) {
            return CompletableFuture.failedFuture((Throwable)new NodeStoppingException());
        }
        LOG.debug("Going to create index [schema={}, table={}, index={}]", new Object[]{schemaName, tableName, indexName});
        try {
            this.validateName(indexName);
            CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
            AtomicBoolean idxExist = new AtomicBoolean(false);
            this.tablesCfg.indexes().change(indexListChange -> {
                idxExist.set(false);
                if (indexListChange.get(indexName) != null) {
                    idxExist.set(true);
                    throw new IndexAlreadyExistsException(schemaName, indexName);
                }
                TableConfiguration tableCfg = (TableConfiguration)this.tablesCfg.tables().get(tableName);
                if (tableCfg == null) {
                    throw new TableNotFoundException(schemaName, tableName);
                }
                ExtendedTableConfiguration exTableCfg = (ExtendedTableConfiguration)tableCfg;
                UUID tableId = (UUID)exTableCfg.id().value();
                Consumer<TableIndexChange> chg = indexChange.andThen(c -> c.changeTableId(tableId));
                indexListChange.create(indexName, chg);
            }).whenComplete((index, th) -> {
                if (th != null) {
                    LOG.debug("Unable to create index [schema={}, table={}, index={}]", th, new Object[]{schemaName, tableName, indexName});
                    if (!failIfExists && idxExist.get()) {
                        future.complete(false);
                    } else {
                        future.completeExceptionally((Throwable)th);
                    }
                } else {
                    TableIndexConfiguration idxCfg = (TableIndexConfiguration)this.tablesCfg.indexes().get(indexName);
                    if (idxCfg != null && idxCfg.value() != null) {
                        LOG.info("Index created [schema={}, table={}, index={}, indexId={}]", new Object[]{schemaName, tableName, indexName, idxCfg.id().value()});
                        future.complete(true);
                    } else {
                        IgniteInternalException exception = new IgniteInternalException(ErrorGroups.Common.UNEXPECTED_ERR, "Looks like the index was concurrently deleted");
                        LOG.info("Unable to create index [schema={}, table={}, index={}]", (Throwable)exception, new Object[]{schemaName, tableName, indexName});
                        future.completeExceptionally((Throwable)exception);
                    }
                }
            });
            CompletableFuture<Boolean> completableFuture = future;
            return completableFuture;
        }
        catch (Exception ex) {
            CompletableFuture<Boolean> completableFuture = CompletableFuture.failedFuture(ex);
            return completableFuture;
        }
        finally {
            this.busyLock.leaveBusy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<Boolean> dropIndexAsync(String schemaName, String indexName, boolean failIfNotExists) {
        if (!this.busyLock.enterBusy()) {
            return CompletableFuture.failedFuture((Throwable)new NodeStoppingException());
        }
        LOG.debug("Going to drop index [schema={}, index={}]", new Object[]{schemaName, indexName});
        try {
            this.validateName(indexName);
            CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
            AtomicBoolean idxOrTblNotExist = new AtomicBoolean(false);
            this.tablesCfg.indexes().change(indexListChange -> {
                idxOrTblNotExist.set(false);
                TableIndexView idxView = (TableIndexView)indexListChange.get(indexName);
                if (idxView == null) {
                    idxOrTblNotExist.set(true);
                    throw new IndexNotFoundException(schemaName, indexName);
                }
                indexListChange.delete(indexName);
            }).whenComplete((ignored, th) -> {
                if (th != null) {
                    LOG.info("Unable to drop index [schema={}, index={}]", th, new Object[]{schemaName, indexName});
                    if (!failIfNotExists && idxOrTblNotExist.get()) {
                        future.complete(false);
                    } else {
                        future.completeExceptionally((Throwable)th);
                    }
                } else {
                    LOG.info("Index dropped [schema={}, index={}]", new Object[]{schemaName, indexName});
                    future.complete(true);
                }
            });
            CompletableFuture<Boolean> completableFuture = future;
            return completableFuture;
        }
        finally {
            this.busyLock.leaveBusy();
        }
    }

    private void validateName(String indexName) {
        if (StringUtils.nullOrEmpty((String)indexName)) {
            throw new IgniteInternalException(ErrorGroups.Index.INVALID_INDEX_DEFINITION_ERR, "Index name should be at least 1 character long");
        }
    }

    private CompletableFuture<?> onIndexDrop(ConfigurationNotificationEvent<TableIndexView> evt) {
        UUID idxId = ((TableIndexView)evt.oldValue()).id();
        if (!this.busyLock.enterBusy()) {
            this.fireEvent(IndexEvent.DROP, new IndexEventParameters(evt.storageRevision(), idxId), (Throwable)new NodeStoppingException());
            return CompletableFuture.failedFuture((Throwable)new NodeStoppingException());
        }
        return ((CompletableFuture)this.tableManager.tableAsync(evt.storageRevision(), ((TableIndexView)evt.oldValue()).tableId()).thenAccept(table -> {
            if (table != null) {
                table.unregisterIndex(idxId);
            }
        })).thenRun(() -> {
            try {
                this.fireEvent(IndexEvent.DROP, new IndexEventParameters(evt.storageRevision(), idxId), null);
            }
            finally {
                this.busyLock.leaveBusy();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<?> onIndexCreate(ConfigurationNotificationEvent<TableIndexView> evt) {
        if (!this.busyLock.enterBusy()) {
            UUID idxId = ((TableIndexView)evt.newValue()).id();
            this.fireEvent(IndexEvent.CREATE, new IndexEventParameters(evt.storageRevision(), idxId), (Throwable)new NodeStoppingException());
            return CompletableFuture.failedFuture((Throwable)new NodeStoppingException());
        }
        try {
            UUID tableId = ((TableIndexView)evt.newValue()).tableId();
            CompletableFuture<?> completableFuture = this.createIndexLocally(evt.storageRevision(), tableId, (TableIndexView)evt.newValue());
            return completableFuture;
        }
        finally {
            this.busyLock.leaveBusy();
        }
    }

    private CompletableFuture<?> createIndexLocally(long causalityToken, UUID tableId, TableIndexView tableIndexView) {
        assert (tableIndexView != null);
        LOG.trace("Creating local index: name={}, id={}, tableId={}, token={}", new Object[]{tableIndexView.name(), tableIndexView.id(), tableId, causalityToken});
        return this.tableManager.tableAsync(causalityToken, tableId).thenAccept(table -> {
            Index<?> index = this.newIndex((TableImpl)table, tableIndexView);
            TableRowToIndexKeyConverter tableRowConverter = new TableRowToIndexKeyConverter(this.schemaManager.schemaRegistry(tableId), ((IndexDescriptor)index.descriptor()).columns().toArray(ArrayUtils.STRING_EMPTY_ARRAY));
            if (index instanceof HashIndex) {
                table.registerHashIndex(tableIndexView.id(), tableIndexView.uniq(), tableRowConverter::convert);
                if (tableIndexView.uniq()) {
                    table.pkId(index.id());
                }
            } else if (index instanceof SortedIndex) {
                table.registerSortedIndex(tableIndexView.id(), tableRowConverter::convert);
            } else {
                throw new AssertionError((Object)("Unknown index type [type=" + index.getClass() + "]"));
            }
            this.fireEvent(IndexEvent.CREATE, new IndexEventParameters(causalityToken, index), null);
        });
    }

    private Index<?> newIndex(TableImpl table, TableIndexView indexView) {
        if (indexView instanceof SortedIndexView) {
            return new SortedIndexImpl(indexView.id(), table, this.convert((SortedIndexView)indexView));
        }
        if (indexView instanceof HashIndexView) {
            return new HashIndex(indexView.id(), table, this.convert((HashIndexView)indexView));
        }
        throw new AssertionError((Object)("Unknown index type [type=" + (indexView != null ? indexView.getClass() : null) + "]"));
    }

    private IndexDescriptor convert(HashIndexView indexView) {
        return new IndexDescriptor(indexView.name(), Arrays.asList(indexView.columnNames()));
    }

    private SortedIndexDescriptor convert(SortedIndexView indexView) {
        int colsCount = indexView.columns().size();
        ArrayList<String> indexedColumns = new ArrayList<String>(colsCount);
        ArrayList<ColumnCollation> collations = new ArrayList<ColumnCollation>(colsCount);
        for (String columnName : indexView.columns().namedListKeys()) {
            IndexColumnView columnView = (IndexColumnView)indexView.columns().get(columnName);
            boolean nullsFirst = !columnView.asc();
            indexedColumns.add(columnName);
            collations.add(ColumnCollation.get(columnView.asc(), nullsFirst));
        }
        return new SortedIndexDescriptor(indexView.name(), indexedColumns, collations);
    }

    private class ConfigurationListener
    implements ConfigurationNamedListListener<TableIndexView> {
        private ConfigurationListener() {
        }

        @NotNull
        public CompletableFuture<?> onCreate(@NotNull ConfigurationNotificationEvent<TableIndexView> ctx) {
            return IndexManager.this.onIndexCreate(ctx);
        }

        @NotNull
        public CompletableFuture<?> onRename(String oldName, String newName, ConfigurationNotificationEvent<TableIndexView> ctx) {
            return CompletableFuture.failedFuture(new UnsupportedOperationException("https://issues.apache.org/jira/browse/IGNITE-16196"));
        }

        @NotNull
        public CompletableFuture<?> onDelete(@NotNull ConfigurationNotificationEvent<TableIndexView> ctx) {
            return IndexManager.this.onIndexDrop(ctx);
        }

        @NotNull
        public CompletableFuture<?> onUpdate(@NotNull ConfigurationNotificationEvent<TableIndexView> ctx) {
            return CompletableFuture.failedFuture(new IllegalStateException("Should not be called"));
        }
    }

    private static class TableRowToIndexKeyConverter {
        private final SchemaRegistry registry;
        private final String[] indexedColumns;
        private final Object mutex = new Object();
        private volatile VersionedConverter converter = new VersionedConverter(-1, t -> null);

        TableRowToIndexKeyConverter(SchemaRegistry registry, String[] indexedColumns) {
            this.registry = registry;
            this.indexedColumns = indexedColumns;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public BinaryTuple convert(BinaryRow tableRow) {
            VersionedConverter converter = this.converter;
            if (converter.version != tableRow.schemaVersion()) {
                Object object = this.mutex;
                synchronized (object) {
                    if (converter.version != tableRow.schemaVersion()) {
                        this.converter = converter = this.createConverter(tableRow.schemaVersion());
                    }
                }
            }
            return converter.convert(tableRow);
        }

        private VersionedConverter createConverter(int schemaVersion) {
            SchemaDescriptor descriptor = this.registry.schema();
            if (descriptor.version() < schemaVersion) {
                this.registry.waitLatestSchema();
            }
            if (descriptor.version() != schemaVersion) {
                descriptor = this.registry.schema(schemaVersion);
            }
            int[] indexedColumns = this.resolveColumnIndexes(descriptor);
            BinaryTupleSchema tupleSchema = BinaryTupleSchema.createSchema((SchemaDescriptor)descriptor, (int[])indexedColumns);
            BinaryConverter rowConverter = new BinaryConverter(descriptor, tupleSchema, false);
            return new VersionedConverter(descriptor.version(), arg_0 -> ((BinaryConverter)rowConverter).toTuple(arg_0));
        }

        private int[] resolveColumnIndexes(SchemaDescriptor descriptor) {
            int[] result = new int[this.indexedColumns.length];
            for (int i = 0; i < this.indexedColumns.length; ++i) {
                Column column = descriptor.column(this.indexedColumns[i]);
                assert (column != null) : this.indexedColumns[i];
                result[i] = column.schemaIndex();
            }
            return result;
        }

        private static class VersionedConverter {
            private final int version;
            private final Function<BinaryRow, BinaryTuple> delegate;

            private VersionedConverter(int version, Function<BinaryRow, BinaryTuple> delegate) {
                this.version = version;
                this.delegate = delegate;
            }

            public BinaryTuple convert(BinaryRow binaryRow) {
                return this.delegate.apply(binaryRow);
            }
        }
    }
}

