/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apache.geode.test.compiler;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.tools.DiagnosticCollector;
import javax.tools.JavaCompiler;
import javax.tools.JavaCompiler.CompilationTask;
import javax.tools.JavaFileManager;
import javax.tools.JavaFileObject;
import javax.tools.ToolProvider;

import com.google.common.base.Charsets;
import org.apache.commons.io.FileUtils;

public class InMemoryJavaCompiler {
  private String classpath = System.getProperty("java.class.path");

  public void addToClasspath(File jarFile) {
    classpath += File.pathSeparator + jarFile.getAbsolutePath();
  }

  public List<InMemoryClassFile> compile(File... sourceFiles) {
    List<InMemorySourceFile> sources = Arrays.stream(sourceFiles)
        .map(this::readFileToString)
        .map(InMemorySourceFile::fromSourceCode)
        .collect(toList());

    return compile(sources);
  }

  public List<InMemoryClassFile> compile(String... sourceFileContents) {
    List<InMemorySourceFile> sources = Arrays.stream(sourceFileContents)
        .map(InMemorySourceFile::fromSourceCode)
        .collect(toList());

    return compile(sources);
  }

  public List<InMemoryClassFile> compile(List<InMemorySourceFile> sources) {
    DiagnosticCollector<JavaFileObject> diagnostics = new DiagnosticCollector<>();
    JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
    JavaFileManager systemFileManager = compiler.getStandardFileManager(diagnostics, null, null);
    InMemoryFileManager fileManager = new InMemoryFileManager(systemFileManager);

    List<String> options = new ArrayList<>(asList("-classpath", classpath));

    CompilationTask compilationTask =
        compiler.getTask(null, fileManager, diagnostics, options, null, sources);

    boolean succeeded = compilationTask.call();
    if (!succeeded) {
      throw new RuntimeException(diagnostics.getDiagnostics().toString());
    }

    return fileManager.compiledClasses();
  }

  private String readFileToString(File file) {
    try {
      return FileUtils.readFileToString(file, Charsets.UTF_8);
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }
}
