/*
 *    Licensed to the Apache Software Foundation (ASF) under one
 *    or more contributor license agreements.  See the NOTICE file
 *    distributed with this work for additional information
 *    regarding copyright ownership.  The ASF licenses this file
 *    to you under the Apache License, Version 2.0 (the
 *    "License"); you may not use this file except in compliance
 *    with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing,
 *    software distributed under the License is distributed on an
 *    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *    KIND, either express or implied.  See the License for the
 *    specific language governing permissions and limitations
 *    under the License.
 */
package org.apache.cayenne.dbsync.naming;

import org.apache.cayenne.configuration.ConfigurationNode;
import org.apache.cayenne.map.ObjEntity;

import java.util.Objects;

/**
 * A builder of names for model objects. Ensures that newly generated names do not conflict with the names of siblings
 * under the same parent node. Name generation can be performed based on default base names for each model object type,
 * or with a user-provided base name. Used standalone or in conjunction with {@link ObjectNameGenerator} that implements
 * DB-to-object name mapping conversions. Names generated by {@link ObjectNameGenerator} can be used as "base names" for
 * {@link NameBuilder}.
 *
 * @since 4.0
 */
public class NameBuilder {

    protected ConfigurationNode nodeToName;
    protected ConfigurationNode parent;
    protected String dupesPattern;
    protected String baseName;

    protected NameBuilder(ConfigurationNode nodeToName) {
        this.nodeToName = Objects.requireNonNull(nodeToName);
        this.dupesPattern = "%s%d";
    }

    public static NameBuilder builder(ConfigurationNode node) {
        return new NameBuilder(node);
    }

    public static NameBuilder builder(ConfigurationNode node, ConfigurationNode parent) {
        return new NameBuilder(node).in(parent);
    }

    /**
     * A special builder starter for callback methods. Eventually callback methods will be made into ConfigurationNodes,
     * and we can use regular {@link #builder(ConfigurationNode)} methods to name them.
     */
    // TODO: fold CallbackMethod to org.apache.cayenne.map package and make it a ConfigurationNode
    // then we can use normal API for it... for now have to keep a special one-off method...
    public static NameBuilder builderForCallbackMethod(ObjEntity parent) {
        return new CallbackNameBuilder().in(parent);
    }

    public NameBuilder in(ConfigurationNode parent) {
        this.parent = Objects.requireNonNull(parent);
        return this;
    }

    public NameBuilder dupesPattern(String dupesPattern) {
        this.dupesPattern = Objects.requireNonNull(dupesPattern);
        return this;
    }

    public NameBuilder baseName(String baseName) {
        this.baseName = baseName;
        return this;
    }

    public String name() {
        String baseName = this.baseName != null && this.baseName.length() > 0
                ? this.baseName
                : nodeToName.acceptVisitor(DefaultBaseNameVisitor.INSTANCE);

        String normalizedBaseName = nodeToName.acceptVisitor(new NormalizationVisitor(baseName));
        return nodeToName.acceptVisitor(new DeduplicationVisitor(parent, normalizedBaseName, dupesPattern));
    }
}
