#!/bin/bash
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#

# requires:
#   node js license-checker to generate the list of deps of this project in the format used by brooklyn
#     (https://github.com/davglass/license-checker)
#   jq to process json (https://stedolan.github.io/jq/)
#   node js json2yaml to output yaml for merging (https://www.npmjs.com/package/json2yaml)

if [ "$1" == "--help" ] ; then
  echo Computes NodeJS dependencies and outputs metadata files in `config/license-`
  echo Works in current directory, or takes target directory as single argument.
  exit 0
fi

EXCLUDE_DEPS=brooklyn

BASE_DIR=$(pushd $(dirname $0) > /dev/null ; pwd -P ; popd > /dev/null)

if [ ! -z "$1" ] ; then pushd $1 > /dev/null ; fi
OUTDIR=.

if [ -d $OUTDIR/config ] ; then OUTDIR=$OUTDIR/config ; fi
TEMPFILE=$OUTDIR/license-dependencies.json.tmp

license-checker --production --json --customPath $BASE_DIR/compute-nodejs-license-dependencies.format.json > $TEMPFILE

cp $BASE_DIR/parts/yaml-asf-header $OUTDIR/license-inclusions-binary-autogenerated-js-license-checker.yaml
cat $TEMPFILE | jq 'keys | .[]' | awk '{ print "- id: "$0 }' | \
  grep -v "id: \"${EXCLUDE_DEPS}-" >> $OUTDIR/license-inclusions-binary-autogenerated-js-license-checker.yaml

# a few things to note:
# * we exclude keys with null/empty values in jq
# * if homepage and repo are the same, or homepage is autogen with #readme appended, we drop homepage
#   ("unique" changes the order, which we don't want)
# * json2yaml prepends --- to the file and "  " on every line, which we don't want
cp $BASE_DIR/parts/yaml-asf-header $OUTDIR/license-metadata-autogenerated-js.yaml
cat $TEMPFILE | jq 'to_entries | 
  map({ id: .key } + 
  .value | { 
    id, name: (.name + " (NodeJS module)"), version, description, 
    url: ([ (select(.homepage != .repository+"#readme" and .homepage != .repository) | .homepage), .repository ]
      | map(select(length > 0))),
    organization: (. | { name: .publisher, email: .email, url: .url }  | with_entries(select((.value | length) > 0)) ),
    license: .licenses,
    copyright_by: .copyright, license_text: .licenseText } 
  | with_entries(select((.value | length) > 0)) )' \
  | json2yaml | grep -v ^--- | sed 's/^  //' \
  >> $OUTDIR/license-metadata-autogenerated-js.yaml

rm $TEMPFILE

echo Reported $(wc $OUTDIR/license-inclusions-binary-autogenerated-js-license-checker.yaml | awk '{print $1}') JS dependencies for $(basename $(pwd))

if [ ! -z "$1" ] ; then popd > /dev/null ; fi

